package common.base.tools.statistics;

import com.google.common.collect.Maps;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.LinkedBlockingQueue;

import org.springframework.http.HttpStatus;

public class ApiQpsStatisticsTools {

    private static Map<ServiceStatisticItem, QpsItem> visitMap = new ConcurrentHashMap();

    private static Map<Integer, ServiceStatisticItem> serverStatisticMap = Collections.synchronizedMap(new HashMap<>());

    private static Map<ServiceStatisticItem, Boolean> lock = new ConcurrentHashMap();

    private static LinkedBlockingQueue<ApiRequestInfo> requestSummary = new LinkedBlockingQueue();

    public static void addVisits(ApiRequestInfo apiRequestInfo) {
        ServiceStatisticItem statisticItem = new ServiceStatisticItem();

        statisticItem.setApiName(apiRequestInfo.getApiName());
        statisticItem.setServerAddress(apiRequestInfo.getServerAddress());
        statisticItem.setServerPort(apiRequestInfo.getServerPort());
        statisticItem.setHttpResponseCode(apiRequestInfo.getHttpResponseCode());
        statisticItem.setErrCode(apiRequestInfo.getErrCode());
        statisticItem.setErrMsg(apiRequestInfo.getErrMsg());
        statisticItem.setArriveTime(apiRequestInfo.getArriveTime());
        int hashCode = statisticItem.hashCode();
        ServiceStatisticItem statistic = serverStatisticMap.putIfAbsent(hashCode, statisticItem);
        if (statistic == null) {
            statistic = statisticItem;
        }
        QpsItem qpsItem = visitMap.computeIfAbsent(statistic, s -> new QpsItem());
        lock.putIfAbsent(statistic, new Boolean(true));

        updateVisitInfo(statistic, qpsItem, apiRequestInfo.getDuration());
    }

    private static void updateVisitInfo(ServiceStatisticItem statisticItem, QpsItem qpsItem, long duration) {
            qpsItem.setVisitCount(qpsItem.getVisitCount() + 1);
            if (String.valueOf(HttpStatus.OK.value()).equals(statisticItem.getHttpResponseCode())) {
                qpsItem.setArriveTime(Math.min(qpsItem.getArriveTime(), statisticItem.getArriveTime()));
                qpsItem.setMaxDurationMs(Math.max(qpsItem.getMaxDurationMs(), duration));
                qpsItem.setMinDurationMs(Math.min(qpsItem.getMinDurationMs(), duration));
                qpsItem.setAvgDurationMs(qpsItem.getAvgDurationMs() + (duration - qpsItem.getAvgDurationMs()) / qpsItem.getVisitCount());
            }
    }

    public static Map<ServiceStatisticItem, QpsItem> getAndResetVisit() {
        Map<ServiceStatisticItem, QpsItem> old = visitMap;
        visitMap = Maps.newConcurrentMap();
        serverStatisticMap = Maps.newConcurrentMap();
        return old;
    }

    public static void statisticRequests(String apiName, String serverAddress, String serverPort,
                                         String httpResponseCode, int errCode, String errMsg, long duration) {
        ApiRequestInfo apiRequestInfo = new ApiRequestInfo();
        apiRequestInfo.setApiName(apiName);
        apiRequestInfo.setServerAddress(serverAddress);
        apiRequestInfo.setServerPort(serverPort);
        apiRequestInfo.setHttpResponseCode(httpResponseCode);
        apiRequestInfo.setErrCode(errCode);
        apiRequestInfo.setErrMsg(errMsg);
        apiRequestInfo.setDuration(duration);
        apiRequestInfo.setArriveTime(System.currentTimeMillis());
        requestSummary.add(apiRequestInfo);
    }

    public static List<ApiRequestInfo> getRequestSummary() {
        List<ApiRequestInfo> list = new ArrayList<>();
        requestSummary.drainTo(list);
        return list;
    }
}
