package common.base.tools.consistenthash;

import common.base.tools.consistenthash.hashing.NativeHash;

import java.util.Collection;
import java.util.Objects;
import java.util.SortedMap;
import java.util.TreeMap;

/**
 * This is an implementation of a consistent hash. T is the type of a Object
 *
 * @param <T>
 * @author Frank.Huang
 */
public class ConsistentHash<T extends INodeObject> {
    /**
     * Of times a bin is replicated in hash circle. (for better load balancing)
     */

    /**
     * Virtual node count,default is 160
     */
    private final int virtualNodeCount;
    private final IHashFunction hashFunction;
    private final SortedMap<Long, T> circle = new TreeMap<Long, T>();

    public final static int VIRTUAL_NODE_COUNT = 160;

    public ConsistentHash(Collection<T> nodes, int virtualNodeCount, IHashFunction IHashFunction) {
        this.virtualNodeCount = virtualNodeCount;
        this.hashFunction = IHashFunction;
        nodes.forEach(e -> add(e));
    }

    public ConsistentHash(Collection<T> nodes, int virtualNodeCount) {
        this(nodes, virtualNodeCount, new NativeHash());
    }

    public ConsistentHash(Collection<T> nodes) {
        this(nodes, VIRTUAL_NODE_COUNT, new NativeHash());
    }


    /**
     * Add node
     *
     * @param node
     */
    public void add(T node) {
        if (Objects.nonNull(node)) {
            for (int i = 0; i < virtualNodeCount; i++) {
                // The string addition forces each replica to have different hash
                circle.put(hashFunction.hash(buildVirtualNodeKey(node.getKey(), i)), node);
            }
        }
    }

    /**
     * Remove node
     *
     * @param node
     */
    public void remove(T node) {
        if (Objects.nonNull(node)) {
            for (int i = 0; i < virtualNodeCount; i++) {
                // The string addition forces each replica to have different hash
                circle.remove(hashFunction.hash(buildVirtualNodeKey(node.getKey(), i)));
            }
        }
    }

    /**
     * This returns the closest node for the object. If the object is the node it
     * should be an exact hit, but if it is a value traverse to find closest
     * subsequent node.
     */
    public T get(String key) {
        if (circle.isEmpty()) {
            return null;
        }

        //object is the node it should be an exact hit
        long hash = hashFunction.hash(key);
        T node = circle.get(hash);
        if (Objects.nonNull(node)) {
            return node;
        }

        //traverse to find closest subsequent node
        SortedMap<Long, T> tailMap = circle.tailMap(hash);
        hash = tailMap.isEmpty() ? circle.firstKey() : tailMap.firstKey();
        node = circle.get(hash);
        return node;
    }

    private String buildVirtualNodeKey(String key, int index) {
        return String.format("%s.%d", key, index);
    }
}
