package common.base.tools.limiter;

import com.google.common.cache.Cache;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractRateLimiter<K, V> {
    private static final Logger logger = LoggerFactory.getLogger(AbstractRateLimiter.class);

    protected Cache<K, V> limitCache = null;
    protected String limiterName = "";
    protected int timeUnitSeconds = 0;
    protected int maxPerTimeUnit = 0;


    public AbstractRateLimiter(String limiterName, int timeUnitSeconds, int maxPerTimeUnit) {
        this.limiterName = limiterName;
        this.timeUnitSeconds = timeUnitSeconds;
        this.maxPerTimeUnit = maxPerTimeUnit;

        limitCache = RateLimiterCacheTools.createCache(limiterName, timeUnitSeconds);

        logger.info("[{}]Create limiter TimeUnit:[{}] Limit:[{}]", limiterName, timeUnitSeconds, maxPerTimeUnit);
    }


    public abstract long incAndGetCount(K key, V value);

    public abstract K getKey(K key);

    public abstract long getCount(V count);

    public long addBlock(K key, V value) {
        K limitKey = getKey(key);
        V current = limitCache.getIfPresent(limitKey);
        if (current == null) {
            current = value;
        }

        long count = incAndGetCount(key, current);
        logger.info("[{}]Add key[{}] count[{}]", limiterName, key, count);
        return count;
    }

    /**
     * check is allow or not
     *
     * @param key
     * @return
     */
    public boolean isOutOfLimit(K key, boolean justCheck) {
        K limitKey = getKey(key);
        V current = limitCache.getIfPresent(limitKey);

        long currentCount = 0;
        if (justCheck) {
            currentCount = getCount(current);
        } else {
            currentCount = incAndGetCount(limitKey, current);
        }

        if (currentCount >= maxPerTimeUnit) {
            //Only print info when first time limit
            if (currentCount == maxPerTimeUnit) {
                logger.info("[{}]OOL.key[{}] Limit[{}]", limiterName, key, maxPerTimeUnit);
            }

            if (justCheck) {
                //if only check, if block add count
                incAndGetCount(limitKey, current);
            }

            return true;
        }

        return false;
    }

    public boolean isOutOfLimit(K key) {
        return isOutOfLimit(key, false);
    }
}
