package common.base.tools.token.serializer;

import com.google.common.collect.Lists;

import java.nio.ByteBuffer;
import java.util.List;

public class ByteBufferBuilder {
    private final byte TYPE_INT = 1;
    private final byte TYPE_LONG = 2;
    private final byte TYPE_STRING = 3;

    ByteBuffer buffer;
    int capacity = -1;

    public ByteBufferBuilder() {
        this(1024 * 1024);
    }


    public ByteBufferBuilder(int capacity) {
        if (capacity <= 0) {
            this.capacity = 1024 * 1024; //default is 1k
        } else {
            this.capacity = capacity;
        }

        buffer = ByteBuffer.allocate(this.capacity);
    }

    public ByteBufferBuilder(byte[] bytes) {
        buffer = ByteBuffer.wrap(bytes);
    }


    public ByteBufferBuilder putInt(String key, int value) {
        putString(key);
        putInt(value);
        return this;
    }

    public ByteBufferBuilder putLong(String key, long value) {
        putString(key);
        putLong(value);
        return this;
    }

    public ByteBufferBuilder putString(String key, String value) {
        putString(key);
        putString(value);
        return this;
    }

    public ByteBufferBuilder put(String key, Object value) {
        if (value instanceof Integer) {
            putInt(key, (Integer) value);
        } else if (value instanceof Long) {
            putLong(key, (Long) value);
        } else if (value instanceof String) {
            putString(key, (String) value);
        } else {
            throw new IllegalArgumentException(String.format("Unsupported Key[{}] type[{}]", key, value.getClass()));
        }
        return this;
    }

    public ByteBufferBuilder putString(String value) {
        buffer.put(TYPE_STRING);
        buffer.putShort((short) value.length());
        buffer.put(value.getBytes());
        return this;
    }

    public ByteBufferBuilder putInt(int value) {
        buffer.put(TYPE_INT);
        buffer.putInt(value);
        return this;
    }

    public ByteBufferBuilder putLong(long value) {
        buffer.put(TYPE_LONG);
        buffer.putLong(value);
        return this;
    }

    public byte[] getAndReset() {
        int len = buffer.position();
        buffer.flip();
        byte[] value = new byte[len];
        buffer.get(value, 0, value.length);
        return value;
    }

    public Object getObject() {
        if (!buffer.hasRemaining()) {
            return null;
        }

        byte type = buffer.get();
        if (type == TYPE_INT) {
            return buffer.getInt();
        } else if (type == TYPE_LONG) {
            return buffer.getLong();
        } else if (type == TYPE_STRING) {
            short len = buffer.getShort();
            byte[] value = new byte[len];
            buffer.get(value);
            return new String(value);
        } else {
            return null;
        }
    }

    public List readToList() {
        List<Object> valueList = Lists.newArrayList();
        Object value = null;
        while (true) {
            value = getObject();
            if (value != null) {
                valueList.add(value);
            } else {
                break;
            }
        }

        return valueList;
    }


}
