package common.base.tools.statistics;

import com.google.common.collect.Maps;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

import org.springframework.http.HttpStatus;

public class ApiQpsStatisticsTools {

    private static final Logger logger = LoggerFactory.getLogger(ApiQpsStatisticsTools.class);

    private static Map<ServiceStatisticItem, QpsItem> visitMap = new ConcurrentHashMap();

    private static Map<Integer, ServiceStatisticItem> serverStatisticMap = Collections.synchronizedMap(new HashMap<>());

    private static Map<ServiceStatisticItem, Boolean> lock = new ConcurrentHashMap();

    public static void addVisits(String apiName, String serverAddress, String serverPort, String httpResponseCode,
                                 String errCode, String errMsg, long duration) {
        ServiceStatisticItem statisticItem = new ServiceStatisticItem(apiName, serverAddress, serverPort,
                httpResponseCode, errCode, errMsg);
        int hashCode = statisticItem.hashCode();
        ServiceStatisticItem statistic = serverStatisticMap.putIfAbsent(hashCode, statisticItem);
        if (statistic == null) {
            statistic = statisticItem;
        }
        QpsItem qpsItem = visitMap.computeIfAbsent(statistic, s -> new QpsItem());
        lock.putIfAbsent(statistic, new Boolean(true));
        updateVisitInfo(statistic, qpsItem, duration);

    }

    private static void updateVisitInfo(ServiceStatisticItem statisticItem, QpsItem qpsItem, long duration) {
        synchronized (lock.get(statisticItem)) {
            String threadName = Thread.currentThread().getName();
            logger.info("------------threadName :[{}] start-----------", threadName);
            qpsItem.setVisitCount(qpsItem.getVisitCount() + 1);
            if (String.valueOf(HttpStatus.OK.value()).equals(statisticItem.getHttpResponseCode())) {
                qpsItem.setMaxDurationMs(Math.max(qpsItem.getMaxDurationMs(), duration));
                qpsItem.setMinDurationMs(Math.min(qpsItem.getMinDurationMs(), duration));
                qpsItem.setAvgDurationMs(qpsItem.getAvgDurationMs() + (duration - qpsItem.getAvgDurationMs()) / qpsItem.getVisitCount());
            }
            logger.info("------------threadName :[{}] end--------------", threadName);
        }
    }

    public static Map<ServiceStatisticItem, QpsItem> getAndResetVisit() {
        Map<ServiceStatisticItem, QpsItem> old = visitMap;
        visitMap = Maps.newConcurrentMap();
        return old;
    }
}
