package common.base.tools.token;

import com.google.common.base.Strings;
import comm.base.tools.api.exception.ApiException;
import common.base.tools.encrypt.AESUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Map;
import java.util.Objects;

public class TokenTools {
    private static final Logger logger = LoggerFactory.getLogger(TokenTools.class);
    private final static int ERR_TOKEN_ILLEGAL = 10201;
    private final static int ERR_TOKEN_EXPIRE = 10202;
    private final static int ERR_TOKEN_VALUE_MISMATCH = 10203;
    private final static int ERR_TOKEN_VALUE_MISSING = 10204;


    public static String encodeToken(String key, TokenObject tokenObject) {
        String tokenStr = tokenObject.serialization();
        try {
            String encodeKey = AESUtils.generateKey(key);
            if (!Strings.isNullOrEmpty(encodeKey)) {
                return AESUtils.encrypt(tokenStr, encodeKey);
            }
        } catch (Exception e) {
            logger.error("[TokenTools] build exception[{}]", tokenObject, e);
        }
        return "";
    }


    public static TokenObject decodeToken(String key, String tokenStr) throws ApiException {
        try {
            String encodeKey = AESUtils.generateKey(key);
            if (!Strings.isNullOrEmpty(encodeKey)) {
                String decodeStr = AESUtils.decrypt(tokenStr, encodeKey);
                TokenObject tokenObject = new TokenObject();
                return tokenObject.deserialization(decodeStr);
            }
        } catch (Exception e) {
            logger.error("[TOKEN]TOKEN_ILLEGAL:[{}]", tokenStr, e);
            throw new ApiException(ERR_TOKEN_ILLEGAL, "TOKEN_ILLEGAL");
        }

        return null;
    }

    public static boolean checkToken(String key, String tokenStr, Map<String, Object> checkValueMap) throws ApiException {
        TokenObject tokenObject = decodeToken(key, tokenStr);
        if (tokenObject == null) {
            return false;
        }

        return checkToken(tokenObject, checkValueMap);
    }


    public static boolean checkToken(TokenObject tokenObject, Map<String, Object> checkValueMap) throws ApiException {
        //check expire time
        long tokenExpireTime = tokenObject.getExpireTime();
        if (tokenExpireTime > 0) {
            if (tokenExpireTime < System.currentTimeMillis()) {
                logger.error("[TOKEN]TOKEN_EXPIRE:[{}]", tokenObject.toString());
                throw new ApiException(ERR_TOKEN_EXPIRE, "TOKEN_EXPIRE");
            }
        }

        for (String key : checkValueMap.keySet()) {
            Object value = checkValueMap.get(key);
            if (Objects.isNull(value)) {
                continue;
            }

            Object tokenValue = tokenObject.getValue(key);
            if (Objects.isNull(tokenValue)) {
                logger.error("[TOKEN]TOKEN_VALUE_MISSING:[{}] key[{}]", tokenObject.toString(), key);
                throw new ApiException(ERR_TOKEN_VALUE_MISSING, "TOKEN_VALUE_MISSING");
            }

            if (value.getClass() != tokenValue.getClass()) {
                logger.error("[TOKEN]TOKEN_VALUE_MISMATCH:[{}] key[{}][{}<>{}]", tokenObject.toString(), key, value.getClass(), tokenValue.getClass());
                throw new ApiException(ERR_TOKEN_VALUE_MISMATCH, "TOKEN_VALUE_MISMATCH");
            }

            boolean checkValue = false;
            if (value instanceof Long) {
                if (((Long) value).compareTo((Long) tokenValue) == 0) {
                    checkValue = true;
                }
            } else if (value instanceof String) {
                if (((String) value).contentEquals((String) tokenValue)) {
                    checkValue = true;
                }
            } else {
                if (value.equals(tokenValue)) {
                    checkValue = true;
                }
            }

            if (!checkValue) {
                logger.error("[TOKEN]TOKEN_VALUE_MISMATCH:[{}] key[{}][{}<>{}]", tokenObject.toString(), key, value, tokenValue);
                throw new ApiException(ERR_TOKEN_VALUE_MISMATCH, "TOKEN_VALUE_MISMATCH");
            }
        }

        return true;
    }
}
