package common.base.tools.statistics;

import com.google.common.base.Strings;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import common.base.tools.sys.SystemTools;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * Created by Frank.Huang on 2016/6/3.
 */
public class TimeStatisticsTools2 {
    public static final String LINE_SEPARATOR = System.getProperty("line.separator");
    public static final long TIME_MAX_NOT_OVER = 180L;

    private static final Logger logger = LoggerFactory.getLogger(TimeStatisticsTools2.class);
    //begin cache map
    public static final Cache<String, Long> timeStartMap = CacheBuilder.newBuilder().expireAfterWrite(TIME_MAX_NOT_OVER, TimeUnit.SECONDS).build();
    //time stat map
    public static final Map<String, StatTimeItem> timeStatMap = Maps.newConcurrentMap();

    public static void addTime(String key, int count, long time) {
        if (Strings.isNullOrEmpty(key)) {
            return;
        }

        StatTimeItem item = timeStatMap.get(key);
        if (item == null) {
            item = new StatTimeItem();
            item.setKey(key);
        }

        item.addCount(count);
        item.addTime(time);

        timeStatMap.put(key, item);
    }

    public static void startTimer(String key) {
        String timeKey = getStatKey(key);
        timeStartMap.put(timeKey, System.currentTimeMillis());
    }

    public static long stopTimer(String key) {
        return stopTimer(key, true);
    }

    public static long stopTimer(String key, boolean isStat) {
        String timeKey = getStatKey(key);
        Long timeStart = timeStartMap.getIfPresent(timeKey);
        if (timeStart == null) {
            return 0;
        }

        long time = System.currentTimeMillis() - timeStart;
        timeStartMap.invalidate(timeKey);

        if (isStat) {
            addTime(key, 1, time);
        }
        return time;
    }

    private static String getStatKey(String key) {
        return String.valueOf(SystemTools.currentThreadId()) + "-" + key;
    }

    public static String dump(boolean printPretty) {
        StringBuilder sb = new StringBuilder();
        if (!timeStatMap.isEmpty()) {
            List<Map.Entry<String, StatTimeItem>> statList = Lists.newArrayList(timeStatMap.entrySet());
            Collections.sort(statList, new Comparator<Map.Entry<String, StatTimeItem>>() {
                @Override
                public int compare(Map.Entry<String, StatTimeItem> o1, Map.Entry<String, StatTimeItem> o2) {
                    return o1.getKey().compareTo(o2.getKey());
                }
            });

            sb.append(printPretty ? LINE_SEPARATOR : "\t");
            sb.append("Time:");
            sb.append(printPretty ? LINE_SEPARATOR : "\t");
            statList.forEach(e -> {
                sb.append("\t").append(Strings.padEnd(e.getKey(), 30, ' '));
                sb.append("count:").append(format(e.getValue().getCount()));
                sb.append("avg:").append(format(e.getValue().avg()));
                sb.append("range:").append(e.getValue().rangeStat());
                sb.append(printPretty ? LINE_SEPARATOR : "\t");
            });
        }

        return sb.toString();
    }

    public static void clear() {
        timeStatMap.clear();
    }

    private static String format(long v) {
        return Strings.padEnd(String.valueOf(v), 6, ' ');
    }

    public static Map<String, StatTimeItem> getTimeStatMap() {
        return timeStatMap;
    }

}
