package common.base.tools.uniqueID;

import com.google.common.base.Strings;
import common.base.tools.sys.SystemTools;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Random;
import java.util.UUID;

/**
 * snowflake的结构如下(每部分用-分开):
 * <p>
 * 1位标识部分    -      41位时间戳部分        -         10位节点部分     12位序列号部分
 * 0 - 0000000000 0000000000 0000000000 0000000000 0 - 00000 - 00000 - 000000000000
 * 第一位为未使用，接下来的41位为毫秒级时间(41位的长度可以使用69年)，然后是5位datacenterId和5位workerId(10位的长度最多支持部署1024个节点） ，最后12位是毫秒内的计数（12位的计数顺序号支持每个节点每毫秒产生4096个ID序号）
 * <p>
 * 一共加起来刚好64位，为一个Long型。(转换成字符串长度为18)
 * <p>
 * snowflake生成的ID整体上按照时间自增排序，并且整个分布式系统内不会产生ID碰撞（由datacenter和workerId作区分），并且效率较高。据说：snowflake每秒能够产生26万个ID。
 * <p>
 * <p>
 * https://github.com/twitter-archive/snowflake
 * https://github.com/souyunku/SnowFlake
 * <p>
 * Usage :
 * 1.Get instance
 * UniqueIdGenerator uniqueIdGenerator = UniqueIdGenerator.getInstance(-1,-1);
 * 2.Call method to generator new id
 * uniqueIdGenerator.nextId()
 * <p>
 * Example：
 * UniqueIdGenerator uniqueIdGenerator = UniqueIdGenerator.getInstance(-1,-1);
 * <p>
 * long start = System.currentTimeMillis();
 * for (int i = 0; i < 100L; i++) {
 * System.out.println(uniqueIdGenerator.nextId());
 * }
 * System.out.println(System.currentTimeMillis() - start);
 */
public class UniqueIdGenerator {
    private static final Logger logger = LoggerFactory.getLogger(UniqueIdGenerator.class);
    /**
     * 起始的时间戳：2020-09-01 00:00:00
     */
    private final static long START_TIMESTAMP = 1598889600000L;

    /**
     * 每一部分占用的位数
     */

    //序列号占用的位数
    private final static long SEQUENCE_BIT = 12;
    //机器标识占用的位数(0~31)
    private final static long MACHINE_BIT = 5;
    //数据中心占用的位数(0~31)
    private final static long DATACENTER_BIT = 5;

    /**
     * 每一部分的最大值
     */
    private final static long MAX_DATACENTER_NUM = -1L ^ (-1L << DATACENTER_BIT);
    private final static long MAX_MACHINE_NUM = -1L ^ (-1L << MACHINE_BIT);
    private final static long MAX_SEQUENCE = -1L ^ (-1L << SEQUENCE_BIT);

    /**
     * 步长, 1024
     */
    private final static long SEQUENCE_STEP_SIZE = 1024L;
    /**
     * 基础序列号, 每发生一次时钟回拨, basicSequence += stepSize
     */
    private long basicSequence = 0L;

    /**
     * 每一部分向左的位移
     */
    private final static long MACHINE_LEFT = SEQUENCE_BIT;
    private final static long DATACENTER_LEFT = SEQUENCE_BIT + MACHINE_BIT;
    private final static long TIMESTMP_LEFT = DATACENTER_LEFT + DATACENTER_BIT;

    //数据中心
    private long datacenterId;
    //机器标识
    private long machineId;
    //序列号
    private long sequence = 0L;
    //上一次时间戳
    private long lastTimestamp = -1L;

    private static class UniqueIdGeneratorInstance {
        private final static UniqueIdGenerator instance = new UniqueIdGenerator();
    }

    public static UniqueIdGenerator getInstance(long datacenterId, long machineId) {
        UniqueIdGeneratorInstance.instance.setWorkId(datacenterId, machineId);
        return UniqueIdGeneratorInstance.instance;
    }

    private UniqueIdGenerator() {
    }

    /**
     * Set workid info
     */
    public void setWorkId(long datacenterId, long machineId) {
        setDatacenterId(datacenterId);
        setMachineId(machineId);
        logger.info("UniqueIdBuilder:datacenterId:[{}] machineId:[{}]", this.datacenterId, this.machineId);
    }

    private long setMachineId(long machineId) {
        if (machineId > 0 && machineId <= MAX_MACHINE_NUM) {
            this.machineId = machineId;
            return this.machineId;
        }

        String machineStr = getMachineStr();
        this.machineId = Math.abs(machineStr.hashCode() % (int) MAX_MACHINE_NUM);
        logger.info("UniqueIdGenerator:MachineId[{}] MachineStr[{}]", this.machineId, machineStr);
        return this.machineId;
    }

    private long setDatacenterId(long datacenterId) {
        if (datacenterId > 0 && datacenterId <= MAX_DATACENTER_NUM) {
            this.datacenterId = datacenterId;
            return this.datacenterId;
        }

        String dataCenterStr = getDatacenterStr();
        this.datacenterId = Math.abs(dataCenterStr.hashCode() % (int) MAX_DATACENTER_NUM);
        logger.info("UniqueIdGenerator:datacenterId[{}] dataCenterStr[{}]", this.datacenterId, dataCenterStr);
        return datacenterId;
    }

    private String getDatacenterStr() {
        String hostName = Strings.nullToEmpty(SystemTools.getHostname());
        String ipList = Strings.nullToEmpty(SystemTools.getIpAddress());

        String str = "";
        if (Strings.isNullOrEmpty(hostName) && Strings.isNullOrEmpty(ipList)) {
            str = UUID.randomUUID().toString();
        } else {
            str = hostName + "@" + ipList;
        }
        return str;
    }

    private String getMachineStr() {
        int pid = SystemTools.getCurrentPid();
        if (pid == -1) {
            pid = UUID.randomUUID().hashCode();
        }
        return getDatacenterStr() + "@" + pid;
    }

    /**
     * 产生下一个ID
     *
     * @return
     */
    public synchronized long nextId() {
        long currentTimestamp = getNewTimestamp();
        if (currentTimestamp < lastTimestamp) {
            /**
             * 时钟回拨:起始步长增加
             *   限制：每毫秒生成的不会超过1024
             */
            logger.warn("UniqueIdBuilder:Clock moved backwards.currentTimestamp:[{}],lastStmp[{}],Offset[{}]", currentTimestamp, lastTimestamp, currentTimestamp - lastTimestamp);
            basicSequence += SEQUENCE_STEP_SIZE;
            if (basicSequence > MAX_SEQUENCE) {
                basicSequence = 0;
                currentTimestamp = getNextMill();
            }
        }

        if (currentTimestamp == lastTimestamp) {
            //相同毫秒内，序列号自增
            sequence = (sequence + 1) & MAX_SEQUENCE;
            //同一毫秒的序列数已经达到最大
            if (sequence == 0L) {
                currentTimestamp = getNextMill();
            }
        } else {
            //不同毫秒内，序列号置为0
            sequence = basicSequence;
        }

        lastTimestamp = currentTimestamp;

        return (currentTimestamp - START_TIMESTAMP) << TIMESTMP_LEFT //时间戳部分
                | datacenterId << DATACENTER_LEFT       //数据中心部分
                | machineId << MACHINE_LEFT             //机器标识部分
                | sequence;                             //序列号部分
    }

    private long getNextMill() {
        long mill = getNewTimestamp();
        while (mill <= lastTimestamp) {
            mill = getNewTimestamp();
        }
        return mill;
    }

    private long getNewTimestamp() {
        return System.currentTimeMillis();
    }

}
