package common.config.tools.config;

import com.google.common.base.Splitter;
import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;
import java.util.Map;

/**
 * Created by Frank.Huang on 2017/2/27.
 */
public abstract class AbstractConfigTools {
    private static final Logger logger = LoggerFactory.getLogger(AbstractConfigTools.class);

    private static final String SPLIT_STR = ",";

    protected static Map<String, String> configMap = Maps.newLinkedHashMap();
    protected static long interval = 1 * 60 * 1000;
    protected static boolean enableEnv = true;


    public static void enableEnv(boolean enableEnv) {
        AbstractConfigTools.enableEnv = enableEnv;
    }

    public static void setInterval(long interval) {
        AbstractConfigTools.interval = interval;
    }


    /**
     * 获取字符串,如果返回为空,则抛出NullPointerException异常
     *
     * @param key:配置项key
     * @param nonEmpty:是否允许值为空
     * @return
     */
    public static String getString(String key, boolean nonEmpty) {
        String value = configMap.get(key);
        if (value == null && enableEnv) {
            value = System.getenv(key);
        }

        if (nonEmpty && Strings.isNullOrEmpty(key)) {
            throw new NullPointerException(String.format("[%s] is empty", key));
        }

        return Strings.nullToEmpty(value);
    }

    public static String getString(String key) {
        return getString(key, false);
    }

    public static String getNonEmptyString(String key) {
        return getString(key, true);
    }

    public static String getString(String key, String defaultValue) {
        String value = getString(key);
        if (Strings.isNullOrEmpty(value)) {
            return defaultValue;
        } else {
            return value;
        }
    }


    public static List<String> getAsList(String key, String split) {
        String value = getString(key);
        if (Strings.isNullOrEmpty(value)) {
            return Lists.newArrayList();
        }

        try {
            return Splitter.on(split).omitEmptyStrings().splitToList(value);
        } catch (Exception e) {
            return Lists.newArrayList();
        }
    }

    public static List<String> getAsList(String key) {
        return getAsList(key, SPLIT_STR);
    }

    public static Long getLong(String key, Long defaultValue) {
        String strValue = getString(key);
        if (Strings.isNullOrEmpty(strValue)) {
            return defaultValue;
        }
        try {

            return Long.parseLong(strValue);

        } catch (NumberFormatException e) {
            return defaultValue;
        }
    }

    public static Long getLong(String key) {
        return getLong(key, null);
    }


    public static Integer getInt(String key, Integer defaultValue) {
        String strValue = getString(key);
        if (Strings.isNullOrEmpty(strValue)) {
            return defaultValue;
        }

        try {
            return Integer.parseInt(strValue);
        } catch (NumberFormatException e) {
            return defaultValue;
        }
    }

    public static Integer getInt(String key) {
        return getInt(key, null);
    }

    public static boolean getBoolean(String key, boolean defaultValue) {
        String value = getString(key);
        if (Strings.isNullOrEmpty(value)) {
            return defaultValue;
        }

        if ("true".equalsIgnoreCase(value)
            || "yes".equalsIgnoreCase(value)
            || "y".equalsIgnoreCase(value)
            || "1".equalsIgnoreCase(value)) {
            return true;
        }

        return false;
    }

    @Deprecated
    public static boolean getAsBoolean(String key, boolean defaultValue) {
        return getBoolean(key, defaultValue);
    }

    public static Map<String, String> getAsMap(String key, String itemSeparator, String keyValueSeparator) {

        String value = null;
        try {
            value = getString(key);
            if (!Strings.isNullOrEmpty(value)) {
                itemSeparator = Strings.isNullOrEmpty(itemSeparator) ? "&" : itemSeparator;
                keyValueSeparator = Strings.isNullOrEmpty(keyValueSeparator) ? ":" : keyValueSeparator;
                return Splitter.on(itemSeparator).withKeyValueSeparator(keyValueSeparator).split(value);
            }
        } catch (Exception e) {
            logger.error("Splitter [{}-{}] [{}] to map exception", itemSeparator, keyValueSeparator, value, e.getMessage());
        }

        return null;
    }

    public static Map<String, String> getAsMap(String key) {
        return getAsMap(key, "&", "=");
    }

    public static Map<String, String> getAllConfig() {
        return configMap;
    }

    public static String replaceDirectoryPath(String path) {
        if (Strings.isNullOrEmpty(path)) {
            return path;
        }

        return path.replace("${user.home}", FileUtils.getUserDirectoryPath())
            .replace("${java.io.tmpdir}", FileUtils.getTempDirectoryPath());
    }

    @Deprecated
    public static String getConfigAsString(String key, String defaultValue) {
        return getString(key, defaultValue);
    }

    @Deprecated
    public static String getConfigAsString(String key) {
        return getString(key, "");
    }

    @Deprecated
    public static Long getConfigAsLong(String key, Long defaultValue) {
        return getLong(key, defaultValue);
    }

    @Deprecated
    public static Long getConfigAsLong(String key) {
        return getLong(key, null);
    }

    /**
     * Deprecated,Use getInt(String,Integer) for replace
     *
     * @param key:config key
     * @return
     */
    @Deprecated
    public static Integer getConfigAsInt(String key, Integer defaultValue) {
        return getInt(key, defaultValue);
    }

    /**
     * Deprecated,Use getInt(String) for replace
     *
     * @param key:config key
     * @return
     */
    @Deprecated
    public static Integer getConfigAsInt(String key) {
        return getInt(key, null);
    }
}
