package common.config.tools.file.monitor;

import com.google.common.base.Strings;
import com.google.common.collect.Maps;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.filefilter.FileFilterUtils;
import org.apache.commons.io.filefilter.IOFileFilter;
import org.apache.commons.io.monitor.FileAlterationMonitor;
import org.apache.commons.io.monitor.FileAlterationObserver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.util.Map;

/**
 * Created by Frank.Huang on 2017/5/3.
 */
public class SingleFileMonitor {
    private static final Logger logger = LoggerFactory.getLogger(SingleFileMonitor.class);

    private FileAlterationMonitor monitor;
    private long interval = 0;
    private Map<String, FileAlterationObserver> directoryObserverMap = Maps.newLinkedHashMap();

    public SingleFileMonitor(long interval) {
        this.interval = interval;
        if (this.interval > 0) {
            //最小扫描周期为：1s
            if (this.interval < 1000) {
                this.interval = 1000;
            }
            monitor = new FileAlterationMonitor(interval);
        } else {
            //默认10分钟,这类并不会真正监听文件变化,只是start的时候不出错
            monitor = new FileAlterationMonitor(10 * 60 * 1000);
        }
    }
    
    private void load(File fn, IFileChangeListener listener) {
        listener.onFileCreate(fn);
    }
    
    public SingleFileMonitor monitorFile(String fn, IFileChangeListener listener) {
        if (Strings.isNullOrEmpty(fn)){
            return this ;
        }
        
        String path = FilenameUtils.getFullPath(fn);
        String name = FilenameUtils.getName(fn);

        return monitorFile(path, name, listener);
    }

    public SingleFileMonitor monitorFile(String path, String fn, IFileChangeListener listener) {
        if (Strings.isNullOrEmpty(path) || Strings.isNullOrEmpty(fn)) {
            return this;
        }

        load(FileUtils.getFile(path, fn), listener);

        if (interval > 0) {
            String ext = FilenameUtils.getExtension(fn);
            String baseName = FilenameUtils.getName(fn);

            IOFileFilter fileFilter = FileFilterUtils.and(
                    FileFilterUtils.fileFileFilter(),
                    FileFilterUtils.suffixFileFilter(ext),
                    new FileNameFilter(baseName));

            FileAlterationObserver observer = new FileAlterationObserver(FileUtils.getFile(path), fileFilter);
            observer.addListener(listener);
            monitor.addObserver(observer);
        }

        return this;
    }

    public void start() {
        try {
            monitor.start();
        } catch (Exception e) {
            logger.info("Start exception:{}", e);
        }
    }

    public void stop() {
        try {
            monitor.stop();
        } catch (Exception e) {
            logger.info("Stop exception:{}", e);
        }
    }
}
