package common.config.tools.file.monitor;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.monitor.FileAlterationListenerAdaptor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Date;

/**
 * Created by Frank.Huang on 2017/5/5.
 */
public class FileChangeListenerAdaptor extends FileAlterationListenerAdaptor implements IFileChangeListener {
    private static final Logger logger = LoggerFactory.getLogger(FileChangeListenerAdaptor.class);

    protected boolean backupAfterProcess = false;

    public FileChangeListenerAdaptor(boolean backupAfterProcess) {
        this.backupAfterProcess = backupAfterProcess;
    }

    @Override
    public void onFileCreate(File file) {
        try{
            super.onFileCreate(file);
            backupFile(file);
        }catch (Throwable e){
            logger.error("onFileCreate exception.",e);
        }
    }

    @Override
    public void onFileChange(File file) {
        try{
            super.onFileChange(file);
            backupFile(file);
        }catch (Throwable e){
            logger.error("onFileCreate exception.",e);
        }
    }

    @Override
    public void setBackupAfterProcess(boolean isBackupAfterProcess) {
        backupAfterProcess = isBackupAfterProcess;
    }

    public void backupFile(File file){
        if (!file.isFile()){
            return;
        }

        if (!backupAfterProcess){
            return;
        }

        String ts = new SimpleDateFormat("yyyyMMdd-HHmmssE").format(new Date());
        String backupPath = FilenameUtils.getFullPath(file.getName()) + File.pathSeparator + "used";
        String fn = FilenameUtils.getName(file.getName()) + FilenameUtils.EXTENSION_SEPARATOR + ts;

        String backupFn = backupPath + File.pathSeparator + fn;

        logger.info("Backup File src[{}] dst[{}]",file.getName(),backupFn);

        try {
            FileUtils.forceMkdir(FileUtils.getFile(backupPath));
            FileUtils.moveFile(file, FileUtils.getFile(backupFn));
        } catch (IOException e) {
            logger.info("Backup file exception:{}", e);
        }
    }
}
