package common.config.tools.config;

import common.config.tools.config.loader.ConfigToolsConfigLoader;
import common.config.tools.configcenter.ConfigCenterService;
import common.config.tools.config.listener.IConfigChangeListener;
import common.config.tools.configcenter.common.ConfigOverrideTools;
import common.config.tools.configcenter.common.RunMode;
import common.config.tools.file.monitor.FileContentListener;
import common.config.tools.file.monitor.PathFileMonitor;
import common.config.tools.model.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.HashSet;

import static common.config.tools.config.AbstractConfigTools.replaceDirectoryPath;


/**
 * @author pyl
 */
public class ConfigRouteTools {
    private static final Logger logger = LoggerFactory.getLogger(ConfigRouteTools.class);

    /**
     * @param configObj config-tools配置对象
     * @param listener  文件内容变更回调
     *                  <p>
     *                  根据初始目录下config.json文件内容，选择启动方式加载配置文件。
     *                  <p>
     *                  启动方式路由
     */
    public void loadConfig(ConfigToolConfigDTO configObj, IConfigChangeListener listener) {
        long start = System.currentTimeMillis();
        initParameter(configObj);
        switch (configObj.getRunMode()) {
            case RunMode.LOCAL:
                localStart(configObj, listener);
                break;
            case RunMode.SERVER:
                configCenterStart(configObj, listener);
                break;
            case RunMode.MIX:
                //默认先加载本地配置，后加载远端配置 以刷新<k,v>
                localStart(configObj, listener);
                configCenterStart(configObj, listener);
                break;
            default:
                break;
        }
        logger.info("loading finished,runMode:[{}] time consume:[{}]s", configObj.getRunMode(), (System.currentTimeMillis() - start) / 1000.0);
    }

    /**
     * @param configObj 初始化全局变量
     */
    private void initParameter(ConfigToolConfigDTO configObj) {
        //层级
        ConfigOverrideTools.initLevels(configObj.getProject(), configObj.getModule(), configObj.getEnv(), configObj.getRegion());
        //<k,v>文件后缀
        ConfigValueTools.extSet = new HashSet<>(configObj.getPropertiesExtension());
    }

    /**
     * @param configObj
     * @param listener  本地启动
     */
    private void localStart(ConfigToolConfigDTO configObj, IConfigChangeListener listener) {
        LocalConfigDTO localInitInfo = configObj.getLocalConfig();
        new PathFileMonitor(localInitInfo.getRefreshInterval())
                .monitorPath(replaceDirectoryPath(localInitInfo.getPath()), null, null, new FileContentListener(false, listener), localInitInfo.getRecursive())
                .start();
    }


    /**
     * @param configObj
     * @param listener  配置中心服务启动
     */
    private void configCenterStart(ConfigToolConfigDTO configObj, IConfigChangeListener listener) {
        new ConfigCenterService(configObj.getConfigCenterConfig()).load(listener);
    }
}
