package common.config.tools.configcenter.common;

import com.google.common.base.Strings;
import com.google.common.collect.Maps;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;

import static common.config.tools.config.AbstractConfigData.DOT;

public class ConfigOverrideTools {
    private static final Logger logger = LoggerFactory.getLogger(ConfigOverrideTools.class);
    /**
     * dataId = [Project]-[Module]-[environment]-[region]-[ConfigFileName]
     * Project:项目名称,可选
     * Module:模块名称,可选
     * environment:环境，可选
     * region:区域,可选
     * ConfigFileName:配置项文件名
     */
    private static final String NO_LEVELS = "";
    private static Map<Integer, String> levelsMap = Maps.newHashMap();


    /**
     * @param project 项目
     * @param module  某块
     * @param env     环境
     * @param region  区域
     *                <p>
     *                获取{Project} {Module} {environment} {region}数据 初始化层级列表
     */
    public static void initLevels(String project, String module, String env, String region) {
        List<String> list = Arrays.asList(project, module, env, region);
        levelsMap.put(1, NO_LEVELS);
        for (int i = 0; i < list.size(); i++) {
            if (!Strings.isNullOrEmpty(list.get(i)) && levelsMap.size() <= 5) {
                if (i == 0) {
                    levelsMap.put(i + 2, list.get(i));
                } else {
                    levelsMap.put(i + 2, levelsMap.get(i + 1) + DOT + list.get(i));
                }
                logger.info("[Init levels:{},{}th]", list.get(i), i);
            } else {
                break;
            }
        }
    }

    /**
     * @param configFileName 原配置文件名
     * @return <dataId,weight>
     * 其中dataid在获取远端配置时使用，配置中心内的DataId需要与其对应
     * weight,为此dataId在同原文件名文件组中的权重值,权重越大,其内容优先级越高
     * <p>
     * 在初始化层级列表后,以原配置文件名获取各层<文件名,权重>
     */
    public static Map<String, Integer> getDataIdsWithWeights(String configFileName) {
        Map<String, Integer> map = new HashMap<>();
        map.put(configFileName, 1);
        for (int i = 2; i <= levelsMap.size(); i++) {
            if (Objects.nonNull(levelsMap.get(i))) {
                logger.info("[Get DataID:{},weight:{}]", levelsMap.get(i) + DOT + configFileName, i);
                map.put(levelsMap.get(i) + DOT + configFileName, i);
            }
        }
        return map;
    }

    /**
     * @param dataId = levels+configFileName
     * @return weight
     * <p>
     * 传入已组合的文件名,返回其权重,权重越大,其内容优先级越高
     */
    public static int getConfigFileWight(String dataId) {
        for (int i = levelsMap.size(); i > 1; i--) {
            if (Objects.nonNull(levelsMap.get(i)) && dataId.contains(levelsMap.get(i) + DOT)) {
                logger.info("[Judge dataID:{},weight:{}]", dataId, i);
                return i;
            }
        }
        //无层级设置返回原文件名=dataId 与权重1
        logger.info("[Judge DataID:{},weight:{}]", dataId, 1);
        return 1;
    }


    /**
     * @param dataId 层级+文件原名
     * @param weight 权重
     * @return String configFileName
     * <p>
     * 传入的dataId与权重, 获得去除层级的configFileName
     */
    public static String getConfigFileNameFromDataId(String dataId, int weight) {
        if (weight <= 1) {
            return dataId;
        } else {
            return dataId.substring(levelsMap.get(weight).length() + 1);
        }
    }
}

