package common.config.tools.file.monitor;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOCase;
import org.apache.commons.io.filefilter.FileFilterUtils;
import org.apache.commons.io.filefilter.IOFileFilter;
import org.apache.commons.io.filefilter.SuffixFileFilter;
import org.apache.commons.io.monitor.FileAlterationListenerAdaptor;
import org.apache.commons.io.monitor.FileAlterationMonitor;
import org.apache.commons.io.monitor.FileAlterationObserver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.Set;

/**
 * Created by Frank.Huang on 2017/5/3.
 */

public class FileMonitor {
    private static final Logger logger = LoggerFactory.getLogger(FileMonitor.class);
    
    private FileAlterationMonitor monitor;
    private long interval = 0;
    private Map<String, FileAlterationObserver> directoryObserverMap = Maps.newLinkedHashMap();

    public FileMonitor(long interval) {
        this.interval = interval;
        if (this.interval > 0) {
            //最小扫描周期为：1s
            if (this.interval < 1000) {
                this.interval = 1000;
            }
            monitor = new FileAlterationMonitor(interval);
        } else {
            //默认10分钟,这类并不会真正监听文件变化,只是start的时候不出错
            monitor = new FileAlterationMonitor(10 * 60 * 1000);
        }
    }

    public FileMonitor addDirectoryListener(String filePath, Set<String> fileExt, IFileChangeListener listener) {
        return addDirectoryListener(filePath, fileExt, listener, false);
    }

    public FileMonitor addDirectoryListener(String filePath, Set<String> fileExt, IFileChangeListener listener, final boolean recursive) {
        return addDirectoryListener(filePath,new SuffixFileFilter(Lists.newArrayList(fileExt), IOCase.INSENSITIVE),listener,recursive);
    }

    public FileMonitor addDirectoryListener(String filePath, IOFileFilter filter, IFileChangeListener listener, final boolean recursive) {
        File rootDir = FileUtils.getFile(filePath);
        if (!rootDir.isDirectory() || !rootDir.exists()) {
            logger.warn("File [{}] is not Directory or not Exist.", rootDir);
            return this;
        }

        IOFileFilter fileFilter = null;
        if (filter == null) {
            fileFilter = FileFilterUtils.fileFileFilter();
        } else {
            fileFilter = filter;
        }

        //加载数据
        loadOnStartup(listener, rootDir, fileFilter, recursive);

        if(interval > 0){
            FileAlterationObserver observer = getFileAlterationObserver(rootDir, fileFilter, listener);
            monitor.addObserver(observer);
            directoryObserverMap.put(rootDir.getAbsolutePath(),observer);
            logger.info("Add directory[{}] to monitor.", rootDir.getAbsolutePath());

            if (recursive) {
                addSubDirectoryObserver(rootDir, fileFilter, listener);
            }
        }


        return this;
    }

    private void addSubDirectoryObserver(File rootPath, IOFileFilter fileFilter, IFileChangeListener listener) {
        Collection<File> files = FileUtils.listFilesAndDirs(rootPath, FileFilterUtils.directoryFileFilter(), FileFilterUtils.directoryFileFilter());
        if (files != null && files.size() > 0) {
            for (File f : files) {
                if (f.isDirectory()) {
                    if (FilenameUtils.equals(rootPath.getAbsolutePath(),f.getAbsolutePath())){
                        continue;
                    }

                    if (directoryObserverMap.containsKey(f.getAbsolutePath())) {
                        continue;
                    }
                    
                    FileAlterationObserver observer = getFileAlterationObserver(f, fileFilter, listener);
                    monitor.addObserver(observer);
                    directoryObserverMap.put(f.getAbsolutePath(), observer);
                    logger.info("Add dir[{}] to monitor.", f.getAbsolutePath());
                }
            }
        }
    }

    private FileAlterationObserver getFileAlterationObserverDirAdd(File rootPath, IOFileFilter fileFilter, IFileChangeListener listener) {
        FileAlterationObserver dirChanged = new FileAlterationObserver(rootPath, FileFilterUtils.directoryFileFilter());
        dirChanged.addListener(new FileAlterationListenerAdaptor() {
            @Override
            public void onDirectoryChange(File directory) {
                addSubDirectoryObserver(directory, fileFilter, listener);
            }

            @Override
            public void onDirectoryDelete(File directory) {
                FileAlterationObserver observer = directoryObserverMap.get(directory.getAbsolutePath());
                if (observer != null) {
                    monitor.removeObserver(observer);
                }
            }
        });
        return dirChanged;
    }

    private void loadOnStartup(IFileChangeListener listener, File rootPath, IOFileFilter fileFilter, final boolean recursive) {
        //初始load
        Collection<File> files = Collections.EMPTY_SET;
        if (recursive) {
            files = FileUtils.listFiles(rootPath, fileFilter, FileFilterUtils.directoryFileFilter());
        } else {
            files = FileUtils.listFiles(rootPath, fileFilter,null);
        }

        if (files != null) {
            files.forEach(e -> {
                listener.onFileChange(e);
            });
        }
    }

    private FileAlterationObserver getFileAlterationObserver(File rootPath, IOFileFilter fileFilter, IFileChangeListener listener) {
        FileAlterationObserver observer = new FileAlterationObserver(rootPath, fileFilter);
        observer.addListener(listener);
        return observer;
    }

    
    public void start() {
        try {
            monitor.start();
        } catch (Exception e) {
            logger.info("Start exception:{}", e);
        }
    }

    public void stop() {
        try {
            monitor.stop();
        } catch (Exception e) {
            logger.info("Stop exception:{}", e);
        }
    }
}
