package common.config.tools.config;

import com.google.common.base.Strings;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import common.config.tools.config.loader.properties.PropertiesLoader;
import common.config.tools.configcenter.common.ConfigOverrideTools;
import common.config.tools.config.listener.IConfigChangeListener;
import common.config.tools.model.ConfigChangeListenerDTO;
import common.config.tools.model.ConfigValueDTO;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;

public class ConfigValueTools extends AbstractConfigData {
    private static final Logger logger = LoggerFactory.getLogger(ConfigValueTools.class);
    public static Map<String, ConfigChangeListenerDTO> listenerMap = Maps.newHashMap();
    public static Set<String> extSet = Sets.newHashSet();

    public static void registerListener(Map<String, Integer> fileMap, IConfigChangeListener listener) {
        fileMap.forEach((k, v) -> registerListener(k, v, listener));
    }

    public static void registerListener(String fileName, int weight, IConfigChangeListener listener) {
        ConfigChangeListenerDTO listenerDTO = listenerMap.get(fileName);
        if (Objects.isNull(listenerDTO)) {
            listenerDTO = new ConfigChangeListenerDTO(fileName, weight, listener);
        } else {
            int oldWeight = listenerDTO.getWeight();
            listenerDTO.setWeight(weight);
            listenerDTO.setListener(listener);
            logger.info("[{}] register listener change. oldWeight[{}]->newWeight[{}]", fileName, oldWeight, weight);
        }
        listenerMap.put(fileName, listenerDTO);
    }

    public static void update(String dataId, String content) {
        if (Strings.isNullOrEmpty(content)) {
            return;
        }
        updateValue(dataId, ConfigOverrideTools.getConfigFileNameFromDataId(dataId, getListener(dataId).getWeight()), content);
        if (isProperties(dataId)) {
            loader(dataId, content);
        }
    }

    private static void loader(String dataId, String content) {
        Map<String, String> valueMap = PropertiesLoader.loader(content);
        valueMap.forEach((k, v) -> {
            updateValue(dataId, k, v);
        });
    }

    private static void updateValue(String dataId, String key, String value) {
        ConfigChangeListenerDTO listenerDTO = getListener(dataId);
        if (Objects.isNull(listenerDTO)) {
            logger.error("Config listener Missing ,dataId[{}]", dataId);
            return;
        }

        int weight = listenerDTO.getWeight();
        ConfigValueDTO oldValue = null;
        ConfigValueDTO newValue = new ConfigValueDTO(value, weight);

        boolean valueChanged = true;
        if (configMap.containsKey(key)) {
            oldValue = configMap.get(key);
            if (ConfigValueDTO.compare(oldValue, newValue) >= 0) {
                valueChanged = false;
            }
        } else {
            oldValue = ConfigValueDTO.emptyValue();
        }

        if (valueChanged) {
            configMap.put(key, newValue);

            //call listen notice user
            IConfigChangeListener listener = listenerDTO.getListener();
            if (Objects.nonNull(listener)) {
                listener.event(dataId, key, oldValue.getValue(), value);
            }
            logger.info("[{}],property modify:key[{}] value[{}]->[{}] level[{}]->[{}]", dataId, key, oldValue.getValue(), value, oldValue.getWeight(), weight);
        }
    }

    private static boolean isProperties(String dataId) {
        String suffix = dataId.substring(dataId.lastIndexOf(DOT) + 1);
        return extSet.contains(suffix);
    }

    public static ConfigChangeListenerDTO getListener(String fileName) {
        return listenerMap.get(fileName);
    }
}
