package common.config.tools.config.loader;

import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.base.Strings;
import common.config.tools.config.loader.json.JsonSerialize;
import common.config.tools.model.ConfigToolConfigDTO;
import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.util.Objects;

/**
 * 读取配置工具配置文件
 */
public class ConfigToolsConfigLoader {

    private static final Logger logger = LoggerFactory.getLogger(ConfigToolsConfigLoader.class);

    /**
     * @param cfgFilePath 可传入 路径或文件
     * @return 返回config-tools的配置对象
     *
     * 获取初始文件对象
     *
     * 若无此文件将返回本地方式启动的配置对象
     */
    public static ConfigToolConfigDTO getConfigObj(String cfgFilePath) {
        File cfgFile = getConfigObjFile(cfgFilePath);
        if (Objects.isNull(cfgFile)) {
            return new ConfigToolConfigDTO(1, cfgFilePath);
        }
        return getConfigObj(cfgFile);
    }


    /**
     * @param cfgFile
     * @return
     * 1. 当无路径或路径下没有初始文件"config.json"时返回空
     * 2. 传入为文件时返回此文件
     * 3. 传入路径下有指定初始文件时返回此文件
     *
     * 4. 传入错误路径将报错
     */
    private static File getConfigObjFile(String cfgFile) {

        if (Strings.isNullOrEmpty(cfgFile)) {
            return null;
        }

        File root = FileUtils.getFile(cfgFile);
        if (!root.exists()){
            throw new RuntimeException("File is not Exist：" + root);
        }

        if (root.isFile()) {
            return root;
        }

        File configJsonFile = FileUtils.getFile(root , "config.json");
        if (configJsonFile.isFile()) {
            return configJsonFile;
        }

        return null;
    }


    /**
     * @param configToolConfigFile config-tools的配置文件
     * @return 将传入的指定文件或路径下默认文件转为config-tools的配置对象 并校验格式返回
     */
    private static ConfigToolConfigDTO getConfigObj(File configToolConfigFile) {

        ConfigToolConfigDTO configObj = (ConfigToolConfigDTO) JsonSerialize.loadJson(configToolConfigFile, new TypeReference<ConfigToolConfigDTO>() {
        });

        if (configObj == null) {
            logger.error("Config json is wrong");
            throw new RuntimeException("Please check if the format of the config.json file is correct");
        }

        if (!configObj.valid()) {
            throw new RuntimeException("Please check if the format of the config.json file is correct");
        }
        return configObj;
    }
}
