package common.config.tools.configcenter;

import com.alibaba.nacos.api.NacosFactory;
import com.alibaba.nacos.api.config.ConfigService;
import com.alibaba.nacos.api.exception.NacosException;
import com.google.common.base.Strings;
import common.config.tools.configcenter.common.ConfigCenterConstants;
import common.config.tools.config.listener.IConfigChangeListener;
import common.config.tools.config.ConfigValueTools;
import common.config.tools.configcenter.common.ConfigOverrideTools;
import common.config.tools.model.ConfigItemDTO;
import common.config.tools.model.ConfigFileDTO;
import common.config.tools.model.ConfigCenterConfigDTO;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;


public class ConfigCenterService {
    private static final Logger logger = LoggerFactory.getLogger(ConfigCenterService.class);
    private ConfigCenterConfigDTO serverConfig;
    private Properties properties = new Properties();

    public ConfigCenterService(ConfigCenterConfigDTO serverConfig) {
        this.serverConfig = serverConfig;
    }

    /**
     * @param listener
     * @return 根据config.json 内config center内容 拉取配置中心配置，并添加监听与注册回调
     */
    public ConfigCenterService load(IConfigChangeListener listener) {
        initProperties();
        for (ConfigItemDTO namespaceConfig : serverConfig.getConfigItems()) {
            loadConfigByNamespace(listener, namespaceConfig);
        }
        return this;
    }

    private void loadConfigByNamespace(IConfigChangeListener listener, ConfigItemDTO namespaceInfo) {
        properties.put(ConfigCenterConstants.NAMESPACE, namespaceInfo.getNamespaceId());
        logger.info("loading namespace[{}]...", namespaceInfo.getNamespaceId());
        try {
            ConfigService configService = NacosFactory.createConfigService(properties);
            for (ConfigFileDTO configFile : namespaceInfo.getConfigFiles()) {
                Map<String, Integer> configFileMap = ConfigOverrideTools.getDataIdsWithWeights(configFile.getConfigFileName());
                ConfigValueTools.registerListener(configFileMap, listener);

                for (Map.Entry<String, Integer> entry : configFileMap.entrySet()) {
                    loadAndListenerByConfigFile(configService, configFile, entry);
                }
            }
        } catch (NacosException e) {
            logger.warn("Init Namespace error code[{}],msg[{}],namespaceId[{}]", e.getErrCode(), e.getErrMsg(), namespaceInfo.getNamespaceId());
        }
    }

    private void loadAndListenerByConfigFile(ConfigService configService, ConfigFileDTO configFile, Map.Entry<String, Integer> entry) {
        try {
            String content = configService.getConfig(entry.getKey(), configFile.getGroup(), serverConfig.getReadTimeout() * 1000L);
            logger.info("loading config group[{}] dataId[{}] contentSize[{}]", configFile.getGroup(), entry.getKey(), StringUtils.length(content));

            ConfigValueTools.update(entry.getKey(), content);

            configService.addListener(entry.getKey(), configFile.getGroup(), new ConfigCenterListener(entry.getKey()));
        } catch (NacosException e) {
            logger.warn("Get and listener config error code[{}],msg[{}],fileName[{}]", e.getErrCode(), e.getErrMsg(), entry.getKey());
        }
    }

    private void initProperties() {
        if (!Strings.isNullOrEmpty(serverConfig.getLocalCacheDir())) {
            System.setProperty(ConfigCenterConstants.LOCAL_CACHE_PATH, serverConfig.getLocalCacheDir());
        }
        properties.put(ConfigCenterConstants.SERVER_ADDR, serverConfig.getServerHost());
        properties.put(ConfigCenterConstants.CONFIG_LONG_POLL_TIMEOUT, serverConfig.getConfigLongPollTimeout());
        properties.put(ConfigCenterConstants.CONFIG_RETRY_TIME, serverConfig.getConfigRetryTime());
        properties.put(ConfigCenterConstants.MAX_RETRY, serverConfig.getMaxRetry());
        properties.put(ConfigCenterConstants.ENABLE_REMOTE_SYNC_CONFIG, serverConfig.getEnableRemoteSyncConfig());
        properties.put(ConfigCenterConstants.USERNAME, serverConfig.getUsername());
        properties.put(ConfigCenterConstants.PASSWORD, serverConfig.getPassword());
    }

}
