package common.config.tools.file.monitor;

import com.google.common.base.Strings;
import com.google.common.collect.Lists;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.filefilter.DirectoryFileFilter;
import org.apache.commons.io.filefilter.FileFilterUtils;
import org.apache.commons.io.filefilter.IOFileFilter;
import org.apache.commons.io.filefilter.NotFileFilter;
import org.apache.commons.io.filefilter.SuffixFileFilter;
import org.apache.commons.io.filefilter.TrueFileFilter;
import org.apache.commons.io.monitor.FileAlterationMonitor;
import org.apache.commons.io.monitor.FileAlterationObserver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;

/**
 * Created by Frank.Huang on 2017/5/3.
 */
public class PathFileMonitor {
    private static final Logger logger = LoggerFactory.getLogger(PathFileMonitor.class);

    private FileAlterationMonitor monitor;
    private long interval = 0;
    
    public PathFileMonitor(long interval) {
        this.interval = interval;
        if (this.interval > 0) {
            //最小扫描周期为：1s
            if (this.interval < 1000) {
                this.interval = 1000;
            }
            monitor = new FileAlterationMonitor(interval);
        } else {
            //默认10分钟,这类并不会真正监听文件变化,只是start的时候不出错
            monitor = new FileAlterationMonitor(10 * 60 * 1000);
        }
    }

    public PathFileMonitor monitorPath(String filePath, String fileNamePattern, Set<String> extSet, IFileChangeListener listener, final boolean recursive) {
        File root = FileUtils.getFile(filePath);
        if (!root.isDirectory()||!root.exists()){
            logger.warn("File [{}] is not Directory or not Exist.", root);
            return this;
        }
        
        IOFileFilter fileFilter = FileFilterUtils.fileFileFilter();
        if (extSet!= null && extSet.size() > 0){
            fileFilter = FileFilterUtils.and(fileFilter,new SuffixFileFilter(Lists.newArrayList(extSet)));
        }

        if (!Strings.isNullOrEmpty(fileNamePattern)){
            fileFilter = FileFilterUtils.and(fileFilter,new FileNameFilter(fileNamePattern));
        }

        //加载数据
        load(listener, root, fileFilter, recursive);
        
        if (interval > 0){
            List<File> paths = Lists.newArrayList();
            if (recursive){
                Collection<File> subDirs = FileUtils.listFilesAndDirs(root,new NotFileFilter(TrueFileFilter.INSTANCE), DirectoryFileFilter.DIRECTORY);
                paths.addAll(subDirs);
            }else {
                paths.add(root);
            }
            
            for (File dir : paths){
                FileAlterationObserver observer = getFileAlterationObserver(dir, fileFilter, listener);
                monitor.addObserver(observer);
                logger.info("Monitor Directory[{}] Filter[Name:{} Extension:{}].",
                        dir.getAbsolutePath(),fileNamePattern,extSet);
            }
        }
        

        return this;
    }
    
    private void load(IFileChangeListener listener, File rootPath, IOFileFilter fileFilter, final boolean recursive) {
        //初始load
        Collection<File> files = Collections.EMPTY_SET;
        if (recursive) {
            files = FileUtils.listFiles(rootPath, fileFilter, FileFilterUtils.directoryFileFilter());
        } else {
            files = FileUtils.listFiles(rootPath, fileFilter,null);
        }

        if (files != null) {
            files.forEach(e -> {
                listener.onFileChange(e);
            });
        }
    }

    private FileAlterationObserver getFileAlterationObserver(File rootPath, IOFileFilter fileFilter, IFileChangeListener listener) {
        FileAlterationObserver observer = new FileAlterationObserver(rootPath, fileFilter);
        observer.addListener(listener);
        return observer;
    }

    
    public void start() {
        try {
            monitor.start();
        } catch (Exception e) {
            logger.info("Start exception:{}", e);
        }
    }

    public void stop() {
        try {
            monitor.stop();
        } catch (Exception e) {
            logger.info("Stop exception:{}", e);
        }
    }
}
