package common.config.tools.file.monitor;

import com.google.common.collect.Maps;
import common.config.tools.config.loader.properties.PropertiesLoader;
import common.config.tools.model.ConfigValueDTO;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.util.Map;


/**
 * Created by Frank.Huang on 2017/5/4.
 */
public class PropertyFileListener extends FileChangeListenerAdaptor {

    private static final Logger logger = LoggerFactory.getLogger(PropertyFileListener.class);
    private Map<String, ConfigValueDTO> properties = Maps.newLinkedHashMap();

    public PropertyFileListener(boolean backupAfterProcess, Map<String, ConfigValueDTO> properties) {
        super(backupAfterProcess);
        this.properties = properties;
    }

    @Override
    public void onFileCreate(File file) {
        logger.info("Property Config File Create:[{}]", file.getAbsolutePath());
        loadProperty(file);
    }

    @Override
    public void onFileChange(File file) {
        logger.info("Property Config File Changed:[{}]", file.getAbsolutePath());
        loadProperty(file);
    }

    @Override
    public void onFileDelete(File file) {
        logger.info("Property Config File Delete:[{}]", file.getAbsolutePath());
    }

    private void loadProperty(File file) {
        /**
         * 注意:本地文件这里没有对文件优先级进行处理
         */
        try {
            Map<String, String> propertyMap = PropertiesLoader.loader(file);
            propertyMap.forEach((k, v) -> {
                if (properties.containsKey(k)) {
                    ConfigValueDTO oldValue = properties.get(k);
                    if (!oldValue.contentEquals(v)) {
                        properties.put(k, ConfigValueDTO.valueOf(v));
                        logger.info("property modify:key[{}] value[{}]->[{}]", k, oldValue, v);
                    }
                } else {
                    logger.info("property load:key[{}] value[{}]", k, v);
                    properties.put(k, ConfigValueDTO.valueOf(v));
                }
            });
        } catch (IOException e) {
            logger.info("Load properties[{}] exception:{}", file.getAbsolutePath(), e);
        }
    }
}
