package com.valor.common.db.config;


import com.google.common.base.Joiner;
import com.google.common.base.Strings;
import com.zaxxer.hikari.HikariDataSource;
import common.config.tools.config.ConfigAESTools;
import common.config.tools.config.ConfigTools3;
import org.hibernate.SessionFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.orm.hibernate5.HibernateTransactionManager;
import org.springframework.orm.hibernate5.LocalSessionFactoryBean;
import org.springframework.transaction.PlatformTransactionManager;

import javax.sql.DataSource;
import java.util.Objects;
import java.util.Properties;


/**
 * Abstract database Configuration
 */
public abstract class AbstractDBConfig {
    private final Logger logger = LoggerFactory.getLogger(AbstractDBConfig.class);
    protected String cfgNamePrefix = "";
    protected String driverClassName;
    protected String dialect;
    protected String validationQuery;
    protected String hbm2ddlAuto;
    protected boolean showSql;
    protected boolean generateStatistics;
    protected int maxActive;
    protected int minIdle;
    protected long idleTimeout;
    protected int txTimeout;


    public AbstractDBConfig() {
        this.cfgNamePrefix = "";
    }

    public AbstractDBConfig(String dbConfigName) {
        this.cfgNamePrefix = dbConfigName;
    }


    /**
     * Get data source
     *
     * @return
     */
    public DataSource getDataSource() {
        String decryptKey = getStringConfig("db.encrypt.key", "");
        String host = getDBConfig("db.host", decryptKey);
        String dbInst = getDBConfig("db.instance", decryptKey);
        String username = getDBConfig("db.username", decryptKey);
        String password = getDBConfig("db.password", decryptKey);

        if (Strings.isNullOrEmpty(host)
                || Strings.isNullOrEmpty(dbInst)
                || Strings.isNullOrEmpty(username)
                || Strings.isNullOrEmpty(password)) {
            throw new IllegalArgumentException(String.format("Database[%s] does not have the proper configuration", cfgNamePrefix));
        }

        String jdbcTpl = getStringConfig("db.jdbc.tpl", "jdbc:mysql://%s/%s?useUnicode=true&characterEncoding=utf8&autoReconnect=true&useSSL=false&serverTimezone=Asia/Shanghai");
        String dbUrl = String.format(jdbcTpl, host, dbInst);

        //HikariDataSource
        driverClassName = getStringConfig("db.driverClassName", "com.mysql.jdbc.Driver");
        validationQuery = getStringConfig("db.validationQuery", "SELECT 1");
        maxActive = getIntConfig("db.pool.maxActive", 50);
        minIdle = getIntConfig("db.pool.minIdle", 10);
        idleTimeout = getIntConfig("db.pool.IdleTimeout", 60000);
        txTimeout = getIntConfig("db.txTimeout", 1800);

        HikariDataSource datasource = new HikariDataSource();
        datasource.setPoolName(String.format("hikari-pool-%s", Strings.nullToEmpty(cfgNamePrefix)));
        datasource.setJdbcUrl(dbUrl);
        datasource.setUsername(username);
        datasource.setPassword(password);
        datasource.setIdleTimeout(idleTimeout);
        datasource.setDriverClassName(driverClassName);
        datasource.setConnectionInitSql(validationQuery);
        datasource.setMinimumIdle(minIdle);
        datasource.setMaximumPoolSize(maxActive);

        printHikariProperties(datasource);
        return datasource;
    }

    /**
     * Get database properties
     *
     * @return
     */
    public Properties getDBProperties() {
        Properties props = new Properties();

        //hibernate config
        dialect = getStringConfig("db.dialect", "org.hibernate.dialect.MySQL55Dialect");
        hbm2ddlAuto = getStringConfig("db.hbm2ddlAuto", "update");
        showSql = getBoolConfig("db.stat.showSql", false);
        generateStatistics = getBoolConfig("db.stat.generateStatistics", false);

        props.put("hibernate.dialect", dialect);
        props.put("hibernate.show_sql", showSql);
        props.put("hibernate.format_sql", false);
        props.put("hibernate.generate_statistics", generateStatistics);
        props.put("hibernate.hbm2ddl.auto", hbm2ddlAuto);
        props.put("hibernate.connection.isolation", java.sql.Connection.TRANSACTION_READ_COMMITTED);
        props.put("hibernate.use_sql_comments", true);

        props.put("hibernate.cache.use_second_level_cache", false);
        props.put("hibernate.cache.use_query_cache", false);
        props.put("hibernate.connection.CharSet", "utf8");
        props.put("hibernate.connection.characterEncoding", "utf8");
        props.put("hibernate.connection.useUnicode", true);
        props.put("hibernate.autoReconnect", true);

        printHibernateProperties(props);
        return props;

    }

    /**
     * Get session factory
     *
     * @param dataSource
     * @return
     */
    public FactoryBean<SessionFactory> getSessionFactory(DataSource dataSource) {
        LocalSessionFactoryBean sessionFactoryBean = new LocalSessionFactoryBean();

        String scanPackages = getStringConfig("db.scanPackages", "");
        String selfScanPackages = "com.valor.common.db";

        if (getBoolConfig("db.common.properties.enable", true)) {
            scanPackages = Joiner.on(",").skipNulls().join(Strings.emptyToNull(scanPackages), selfScanPackages);
        }

        if (!Strings.isNullOrEmpty(scanPackages)) {
            sessionFactoryBean.setPackagesToScan(scanPackages.split(","));
        }

        sessionFactoryBean.setHibernateProperties(getDBProperties());
        sessionFactoryBean.setDataSource(dataSource);
        return sessionFactoryBean;
    }


    public PlatformTransactionManager getTransactionManager(SessionFactory sessionFactory) {
        HibernateTransactionManager transactionManager = new HibernateTransactionManager(sessionFactory);
        transactionManager.setDefaultTimeout(txTimeout);
        return transactionManager;
    }

    private void printHikariProperties(HikariDataSource datasource) {
        if (Objects.isNull(datasource)) {
            return;
        }

        logger.info("[Hikari]PoolName=[{}]", datasource.getPoolName());
        logger.info("[Hikari]JdbcUrl=[{}]", datasource.getJdbcUrl());
        logger.info("[Hikari]Username=[{}]", datasource.getUsername());
        logger.info("[Hikari]IdleTimeout=[{}]", datasource.getIdleTimeout());
        logger.info("[Hikari]DriverClassName=[{}]", datasource.getDriverClassName());
        logger.info("[Hikari]ConnectionInitSql=[{}]", datasource.getConnectionInitSql());
        logger.info("[Hikari]MinimumIdle=[{}]", datasource.getMinimumIdle());
        logger.info("[Hikari]MaximumPoolSize=[{}]", datasource.getMaximumPoolSize());
        logger.info("[Hikari]Catalog=[{}]", datasource.getCatalog());
        logger.info("[Hikari]ConnectionTimeout=[{}]", datasource.getConnectionTimeout());
        logger.info("[Hikari]ValidationTimeout=[{}]", datasource.getValidationTimeout());
    }

    private void printHibernateProperties(Properties props) {
        if (Objects.isNull(props)) {
            return;
        }

        props.forEach((k, v) -> {
            logger.info("[hibernate]{}=[{}]", k, v);
        });
    }

    /**
     * Get db config value
     *
     * @param configKey
     * @param cryptKey
     * @return
     */
    public String getDBConfig(String configKey, String cryptKey) {
        String configItem = buildConfigKey(cfgNamePrefix, configKey);
        if (Strings.isNullOrEmpty(cryptKey)) {
            return ConfigTools3.getString(configItem);
        } else {
            return ConfigAESTools.getAESString(configItem, cryptKey);
        }
    }

    private String buildConfigKey(String configName, String configItem) {
        if (Strings.isNullOrEmpty(configName)) {
            return configItem;
        } else {
            return String.format("%s.%s", configName, configItem);
        }
    }

    /**
     * Get config
     *
     * @param configItem
     * @param defaultValue
     * @return
     */
    protected String getStringConfig(String configItem, String defaultValue) {
        String key = buildConfigKey(cfgNamePrefix, configItem);
        String value = ConfigTools3.getString(key);

        //Not config use name,use common config
        if (Strings.isNullOrEmpty(value) && !Strings.isNullOrEmpty(cfgNamePrefix)) {
            value = ConfigTools3.getString(configItem);
        }

        if (Strings.isNullOrEmpty(value)) {
            return defaultValue;
        } else {
            return value;
        }
    }

    protected int getIntConfig(String configItem, int defaultValue) {
        String key = buildConfigKey(cfgNamePrefix, configItem);
        Integer value = ConfigTools3.getInt(key);

        //Not config use name,use common config
        if (Objects.isNull(value) && !Strings.isNullOrEmpty(cfgNamePrefix)) {
            value = ConfigTools3.getInt(configItem);
        }

        if (Objects.nonNull(value)) {
            return value;
        } else {
            return defaultValue;
        }
    }

    protected boolean getBoolConfig(String configItem, Boolean defaultValue) {
        String key = buildConfigKey(cfgNamePrefix, configItem);
        Boolean value = ConfigTools3.getBoolean(key);

        //Not config use name,use common config
        if (Objects.isNull(value) && !Strings.isNullOrEmpty(cfgNamePrefix)) {
            value = ConfigTools3.getBoolean(configItem);
        }

        if (Objects.nonNull(value)) {
            return value;
        } else {
            return defaultValue;
        }
    }
}
