package common.ipdb.geoip;

import com.github.davidmoten.geo.GeoHash;
import com.google.common.base.Strings;
import com.google.common.collect.Maps;
import com.maxmind.db.CHMCache;
import com.maxmind.geoip2.DatabaseReader;
import com.maxmind.geoip2.exception.AddressNotFoundException;
import com.maxmind.geoip2.model.AsnResponse;
import com.maxmind.geoip2.model.CityResponse;
import com.maxmind.geoip2.record.*;
import common.ipdb.model.IPASNInfo;
import common.ipdb.model.IPInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.net.InetAddress;
import java.util.Map;

public class GeoIpTools {
    private static final Logger logger = LoggerFactory.getLogger(GeoIpTools.class);
    private static Map<EGeoDBTypes, DatabaseReader> dbReaderMap = Maps.newHashMap();

    static {
        loadFromDefaultPath();
    }

    public static String getDefaultPath() {
        //set default path base os
        String os_name = System.getProperty("os.name").toLowerCase();
        if (os_name.contains("window")) {
            // /operate system : windows
            return "c:/valor/ipdb";
        } else {
            //operate system : *unix
            return "/valor/ipdb";
        }
    }

    public static String getOrDefaultDBPath(EGeoDBTypes types, String dbPath) {
        if (Strings.isNullOrEmpty(dbPath)) {
            dbPath = String.format("%s/%s", getDefaultPath(), types.getName());
        }
        return dbPath;
    }


    /**
     * Load ip database file
     *
     * @param types
     * @param dbName
     */
    public static void loadDB(EGeoDBTypes types, String dbName) {
        try {
            DatabaseReader reader = dbReaderMap.get(types);
            if (reader != null) {
                reader.close();
            }
        } catch (Exception e) {
            logger.info("Reader close exeption:{}", e);
        }
        try {
            dbName = getOrDefaultDBPath(types, dbName);

            // A File object pointing to your GeoIP2 or GeoLite2 database
            File dbFile = new File(dbName);
            if (!dbFile.exists()) {
                logger.info("DB[{}] not exist", dbFile);
            } else {
                // This creates the DatabaseReader object, which should be reused across|lookups.
                DatabaseReader reader = new DatabaseReader.Builder(dbFile).withCache(new CHMCache()).build();
                dbReaderMap.put(types, reader);
                logger.info("DB[{}] is loaded", dbFile);
            }

        } catch (Exception e) {
            logger.error("DB[{}] load failed! ex[{}]", dbName, e);
        }
    }

    /**
     * Load database base default path
     */
    public static void loadFromDefaultPath() {
        for (EGeoDBTypes types : EGeoDBTypes.values()) {
            loadDB(types, "");
        }
    }

    /**
     * Get city info
     *
     * @param ip
     * @return
     */
    public static IPInfo getCityInfo(String ip) {
        try {
            DatabaseReader reader = dbReaderMap.get(EGeoDBTypes.GEOLITE2_CITY);
            if (reader == null) {
                return null;
            }

            InetAddress ipAddress = InetAddress.getByName(ip);
            CityResponse response = reader.city(ipAddress);
            if (response == null) {
                return null;
            }

            IPInfo ipInfo = new IPInfo();
            ipInfo.setIp(ip);

            City city = response.getCity();
            if (city != null) {
                ipInfo.setCity(city.getName());
            }

            Country country = response.getCountry();
            if (country != null) {
                ipInfo.setCountry(Strings.nullToEmpty(country.getIsoCode()).toLowerCase());
            }

            Location location = response.getLocation();
            if (location != null) {
                ipInfo.setLatitude(location.getLatitude());
                ipInfo.setLongitude(location.getLongitude());
                ipInfo.setLocation(String.format("%s,%s", location.getLatitude(), location.getLongitude()));
                String geoHash = GeoHash.encodeHash(location.getLatitude(), location.getLongitude());
                ipInfo.setGeoHash(geoHash);
            }

            Postal postal = response.getPostal();
            if (postal != null) {
                ipInfo.setPostCode(postal.getCode());
            }

            return ipInfo;
        } catch (Exception e) {
            printException(ip, e);
        }

        return null;
    }

    private static void printException(String ip, Exception e) {
        if (e instanceof AddressNotFoundException) {
            logger.warn("IP[{}] Not Found", ip, e.getMessage());
        } else {
            logger.error("IP[{}] message:[{}]", ip, e);
        }
    }

    /**
     * Get AS Info
     *
     * @param ip
     * @return
     */
    public static IPASNInfo getASNInfo(String ip) {
        try {
            DatabaseReader reader = dbReaderMap.get(EGeoDBTypes.GEOLITE2_ASN);
            if (reader == null) {
                return null;
            }

            InetAddress ipAddress = InetAddress.getByName(ip);
            AsnResponse response = reader.asn(ipAddress);
            if (response == null) {
                return null;
            }

            IPASNInfo asnInfo = new IPASNInfo();
            asnInfo.setASN(response.getAutonomousSystemNumber());
            asnInfo.setASO(response.getAutonomousSystemOrganization());

            return asnInfo;
        } catch (Exception e) {
            printException(ip, e);
        }

        return null;
    }

    /**
     * Get Ip address info
     *
     * @param ip
     * @return
     */
    public static IPInfo getIPInfo(String ip) {
        if (Strings.isNullOrEmpty(ip)) {
            return null;
        }

        if (dbReaderMap.isEmpty()) {
            return null;
        }

        try {
            IPInfo info = getCityInfo(ip);
            if (info != null) {
                IPASNInfo asnInfo = getASNInfo(ip);
                if (asnInfo != null) {
                    info.setASN(asnInfo.getASN());
                    info.setASO(asnInfo.getASO());
                }
            }
            return info;
        } catch (Exception e) {
            logger.error("IP[{}] message:[{}]", ip, e.getMessage());
        }
        return null;
    }

    /**
     * Get country info
     *
     * @param ip
     * @return
     */
    public static String getCountry(String ip) {
        IPInfo ipInfo = getCityInfo(ip);
        if (ipInfo == null) {
            return "";
        }

        return Strings.nullToEmpty(ipInfo.getCountry());
    }
}
