package common.ipdb.api;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import common.config.tools.config.ConfigTools3;
import common.ipdb.api.geoip.GroIPApi;
import common.ipdb.model.IPInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Objects;
import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;

public class IPDBTools {
    private static final Logger logger = LoggerFactory.getLogger(IPDBTools.class);
    private static IIPDBApi ipQueryProvider;

    private static Cache<String, IPInfo> ipCache = CacheBuilder.newBuilder()
        .expireAfterAccess(ConfigTools3.getInt("ipdb.cache.expire.time", 12 * 60), TimeUnit.MINUTES) //12 hour
        .maximumSize(ConfigTools3.getInt("ipdb.cache.max.size", 64 * 10000)) // 640K
        .build();

    /**
     * Register IP Query Provider
     *
     * @param iIpQueryProvider:ip query api
     */
    public static void registerQueryProvider(IIPDBApi iIpQueryProvider) {
        if (ipQueryProvider != null) {
            ipQueryProvider = iIpQueryProvider;
        }
    }

    /**
     * Query ip address info
     *
     * @param ip
     * @return
     */
    public static IPInfo get(String ip) {
        try {
            IPInfo ipInfo = ipCache.get(ip, new Callable<IPInfo>() {
                @Override
                public IPInfo call() {
                    if (Objects.isNull(ipQueryProvider)) {
                        ipQueryProvider = new GroIPApi();
                    }
                    return ipQueryProvider.query(ip);
                }
            });

            return ipInfo;
        } catch (Exception e) {
            logger.error("IP[{}] [{}]", ip, e.getMessage());
        }

        return null;
    }
}
