package common.ipdb.api;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import common.config.tools.config.ConfigTools3;
import common.ipdb.api.geoip.GeoIPApi;
import common.ipdb.model.IPInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Objects;
import java.util.concurrent.TimeUnit;

public class IPDBService {
    private static final Logger logger = LoggerFactory.getLogger(IPDBService.class);
    private static IIPDBApi ipQueryProvider;

    /**
     * cache time:12h
     * cache max:640k
     */
    private static Cache<String, IPInfo> ipCache = Caffeine.newBuilder()
            .expireAfterAccess(ConfigTools3.getInt("common.ipdb.cache.expire.time", 12 * 60), TimeUnit.MINUTES)
            .maximumSize(ConfigTools3.getInt("common.ipdb.cache.max.size", 64 * 10000))
            .build();

    /**
     * Register IP Query Provider
     *
     * @param iIpQueryProvider:ip query api
     */
    public static void registerQueryProvider(IIPDBApi iIpQueryProvider) {
        if (ipQueryProvider != null) {
            ipQueryProvider = iIpQueryProvider;
        }
    }

    /**
     * Query ip address info
     *
     * @param ip
     * @return
     */
    public static IPInfo get(String ip) {
        try {
            return ipCache.get(ip, k -> getFromIpdb(k));
        } catch (Exception e) {
            //logger.error("IP[{}] [{}]", ip, e.getMessage());
        }

        return null;
    }

    public static IPInfo getFromIpdb(String ip) {
        if (Objects.isNull(ipQueryProvider)) {
            ipQueryProvider = new GeoIPApi();
        }
        return ipQueryProvider.query(ip);
    }
}
