package common.ipdb.api.geoip;

import com.google.common.collect.Maps;
import common.ipdb.filemanager.FiledTools;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Map;
import java.util.Objects;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

public class GeoIPDBManager {
    private static final Logger logger = LoggerFactory.getLogger(GeoIPDBManager.class);

    private static boolean isRunning = false;
    private static boolean isStartupRunning = false;

    private static ScheduledExecutorService executor = Executors.newScheduledThreadPool(4);

    private static Map<EGeoIPDBType, GeoDownloader> downloaderMap = Maps.newHashMap();

    static {
        for (EGeoIPDBType type : EGeoIPDBType.values()) {
            if (!downloaderMap.containsKey(type)) {
                downloaderMap.putIfAbsent(type, new GeoDownloader(type));
            }
        }
    }

    public static void downloadIfDbNotExists() {
        for (EGeoIPDBType type : EGeoIPDBType.values()) {
            if (!GeoIPConfig.isDbExist(type)) {
                GeoDownloader downloader = downloaderMap.get(type);
                if (Objects.nonNull(downloader)) {
                    downloader.run();
                }
            }
        }
    }

    public static void startAutoUpdate() {
        if (isRunning) {
            return;
        }

        for (EGeoIPDBType type : EGeoIPDBType.values()) {
            GeoDownloader downloader = downloaderMap.get(type);
            if (Objects.nonNull(downloader)) {
                executor.scheduleWithFixedDelay(downloader, 1, GeoIPConfig.updateDuration(), TimeUnit.MINUTES);
                logger.info("Start db [{}] auto update duration[{}m]", type, GeoIPConfig.updateDuration());
            }
        }

        isRunning = true;
    }

    public static void stop() {
        executor.shutdown();
    }

    /**
     * 启动db文件自动更新
     */
    public static void startup() {
        if (isStartupRunning) {
            return;
        }

        FiledTools.mkdirsIfNotExist(GeoIPConfig.dbPath());

        GeoIPDBManager.downloadIfDbNotExists();

        for (EGeoIPDBType type : EGeoIPDBType.values()) {
            GeoIPReaderFactory.load(type);
        }

        GeoIPDBManager.startAutoUpdate();
        isStartupRunning = true;
    }

    public static class GeoDownloader implements Runnable {
        private EGeoIPDBType type;


        public GeoDownloader(EGeoIPDBType type) {
            this.type = type;
        }

        @Override
        public void run() {
            if (GeoIPConfig.dbAutoUpdate() && FiledTools.isNeedUpdate(type)) {

                String url = GeoIPConfig.dbDownloadUrl(type);
                String downloadFilename = GeoIPConfig.getDBDownloadFilename(type);

                FiledTools.download(url, downloadFilename);
                FiledTools.uncompressTarGzip(downloadFilename, GeoIPConfig.dbPath(), type.getSuffix());
            }
            GeoIPReaderFactory.load(type);
        }
    }
}
