package common.ipdb.api.geoip;

import com.google.common.collect.Maps;
import com.maxmind.db.CHMCache;
import com.maxmind.db.Reader;
import com.maxmind.geoip2.DatabaseReader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.Map;

public class GeoIPReaderFactory {
    private static final Logger logger = LoggerFactory.getLogger(GeoIPService.class);
    private static Map<EGeoIPDBType, DatabaseReader> dbReaderMap = Maps.newHashMap();

    public static void load(EGeoIPDBType type) {
        //关闭文件使用
        try {
            DatabaseReader reader = dbReaderMap.get(type);
            if (reader != null) {
                reader.close();
            }
        } catch (Exception e) {
            logger.info("Reader close exeption:{}", e);
        }

        //更新文件
        String dbName = GeoIPConfig.dbFilename(type);
        try {
            updateDbIfNecessary(dbName);

            // A File object pointing to your GeoIP2 or GeoLite2 database
            File dbFile = new File(dbName);
            if (!dbFile.exists()) {
                logger.info("DB[{}] not exist", dbFile);
            } else {
                // This creates the DatabaseReader object, which should be reused across|lookups.

                Reader.FileMode fileMode = Reader.FileMode.MEMORY_MAPPED;
                if (GeoIPConfig.isOsWindows()) {
                    //window MEMORY_MAPPED 替换文件会异常
                    fileMode = Reader.FileMode.MEMORY;
                }
                DatabaseReader reader = new DatabaseReader.Builder(dbFile)
                        .fileMode(fileMode)
                        .withCache(new CHMCache())
                        .build();
                dbReaderMap.put(type, reader);
                logger.info("DB[{}] is loaded", dbFile);
            }

        } catch (Exception e) {
            logger.error("DB[{}] load failed! ex[{}]", dbName, e);
        }
    }

    private static void updateDbIfNecessary(String filename) {
        String backupFilename = GeoIPConfig.getFilenameBackup(filename);
        String upgradeFilename = GeoIPConfig.getFilenameUpgrade(filename);
        try {
            //backup current file
            Path current = Paths.get(filename);
            if (Files.exists(current)) {
                Files.copy(current, Paths.get(backupFilename), StandardCopyOption.REPLACE_EXISTING);
                logger.info("GeoIp db backup [{}]->[{}]", filename, backupFilename);
            }

            //update file
            Path upgrade = Paths.get(upgradeFilename);
            if (Files.exists(upgrade)) {
                Files.move(upgrade, Paths.get(filename), StandardCopyOption.REPLACE_EXISTING);
                logger.info("GeoIp db update [{}]->[{}]", upgrade, filename);
            }
        } catch (Exception e) {
            logger.error("Update db field.Upgrade[{}] current[{}]", upgradeFilename, filename, e);
        }
    }

    public static DatabaseReader getReader(EGeoIPDBType type) {
        return dbReaderMap.get(type);
    }

    public static boolean isEmpty() {
        return dbReaderMap.isEmpty();
    }

}
