package common.ipdb.api.geoip;

import com.github.davidmoten.geo.GeoHash;
import com.google.common.base.Strings;
import com.maxmind.geoip2.DatabaseReader;
import com.maxmind.geoip2.exception.AddressNotFoundException;
import com.maxmind.geoip2.model.AsnResponse;
import com.maxmind.geoip2.model.CityResponse;
import com.maxmind.geoip2.record.*;
import common.ipdb.model.IPASNInfo;
import common.ipdb.model.IPInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.InetAddress;

public class GeoIPService {
    private static final Logger logger = LoggerFactory.getLogger(GeoIPService.class);



    /**
     * Get city info
     *
     * @param ip
     * @return
     */
    public static IPInfo getCityInfo(String ip) {
        try {
            DatabaseReader reader = GeoIPReaderFactory.getReader(EGeoIPDBType.GEOLITE2_CITY);
            if (reader == null) {
                return null;
            }

            InetAddress ipAddress = InetAddress.getByName(ip);
            CityResponse response = reader.city(ipAddress);
            if (response == null) {
                return null;
            }

            IPInfo ipInfo = new IPInfo();
            ipInfo.setIp(ip);

            City city = response.getCity();
            if (city != null) {
                ipInfo.setCity(city.getName());
            }

            Country country = response.getCountry();
            if (country != null) {
                ipInfo.setCountry(Strings.nullToEmpty(country.getIsoCode()).toLowerCase());
            }

            Location location = response.getLocation();
            if (location != null) {
                ipInfo.setLatitude(location.getLatitude());
                ipInfo.setLongitude(location.getLongitude());
                ipInfo.setLocation(String.format("%s,%s", location.getLatitude(), location.getLongitude()));
                String geoHash = GeoHash.encodeHash(location.getLatitude(), location.getLongitude());
                ipInfo.setGeoHash(geoHash);
            }

            Postal postal = response.getPostal();
            if (postal != null) {
                ipInfo.setPostCode(postal.getCode());
            }

            return ipInfo;
        } catch (Exception e) {
            printException(ip, e);
        }

        return null;
    }

    private static void printException(String ip, Exception e) {
        if (e instanceof AddressNotFoundException) {
            logger.debug("IP[{}] Not Found", ip, e.getMessage());
        } else {
            logger.error("IP[{}] message:[{}]", ip, e);
        }
    }

    /**
     * Get AS Info
     *
     * @param ip
     * @return
     */
    public static IPASNInfo getASNInfo(String ip) {
        try {
            DatabaseReader reader = GeoIPReaderFactory.getReader(EGeoIPDBType.GEOLITE2_ASN);
            if (reader == null) {
                return null;
            }

            InetAddress ipAddress = InetAddress.getByName(ip);
            AsnResponse response = reader.asn(ipAddress);
            if (response == null) {
                return null;
            }

            IPASNInfo asnInfo = new IPASNInfo();
            asnInfo.setASN(response.getAutonomousSystemNumber());
            asnInfo.setASO(response.getAutonomousSystemOrganization());

            return asnInfo;
        } catch (Exception e) {
            printException(ip, e);
        }

        return null;
    }

    /**
     * Get Ip address info
     *
     * @param ip
     * @return
     */
    public static IPInfo getIPInfo(String ip) {
        if (Strings.isNullOrEmpty(ip)) {
            return null;
        }

        if (GeoIPReaderFactory.isEmpty()) {
            return null;
        }

        try {
            IPInfo info = getCityInfo(ip);
            if (info != null) {
                IPASNInfo asnInfo = getASNInfo(ip);
                if (asnInfo != null) {
                    info.setASN(asnInfo.getASN());
                    info.setASO(asnInfo.getASO());
                }
            }
            return info;
        } catch (Exception e) {
            logger.error("IP[{}] message:[{}]", ip, e.getMessage());
        }
        return null;
    }

    /**
     * Get country info
     *
     * @param ip
     * @return
     */
    public static String getCountry(String ip) {
        IPInfo ipInfo = getCityInfo(ip);
        if (ipInfo == null) {
            return "";
        }

        return Strings.nullToEmpty(ipInfo.getCountry());
    }
}
