package common.ipdb.filemanager;


import com.google.common.base.Strings;
import com.google.common.io.Files;
import common.ipdb.api.geoip.EGeoIPDBType;
import common.ipdb.api.geoip.GeoIPConfig;
import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;
import org.apache.commons.compress.compressors.gzip.GzipCompressorInputStream;
import org.apache.commons.compress.utils.IOUtils;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.channels.Channels;
import java.nio.channels.FileChannel;
import java.nio.channels.ReadableByteChannel;
import java.text.DateFormat;
import java.util.Date;
import java.util.concurrent.TimeUnit;

public class FiledTools {
    private static Logger logger = LoggerFactory.getLogger(FiledTools.class);

    public static void download(String urlPath, String filename) {
        try {
            logger.info("Start download file[{}] from[{}]", filename, urlPath);
            URL url = new URL(urlPath);
            ReadableByteChannel readableByteChannel = Channels.newChannel(url.openStream());

            FileOutputStream fileOutputStream = new FileOutputStream(filename);
            FileChannel fileChannel = fileOutputStream.getChannel();
            fileChannel.transferFrom(readableByteChannel, 0, Long.MAX_VALUE);
            logger.info("Finish download file[{}] ", filename);
        } catch (MalformedURLException e) {
            logger.error("Download Exception file[{}]", filename, e);
        } catch (IOException e) {
            logger.error("Download IOException file[{}]", filename, e);
        }
    }

    public static void uncompressTarGzip(String sourcePath, String targetPath, String suffix) {
        logger.info("Start uncompress file[{}] to[{}]", sourcePath, targetPath);
        File sourceFile = new File(sourcePath);
        File targetFolder = new File(targetPath);

        // 将 tar 文件解压到 extractPath 目录下
        try (TarArchiveInputStream fin = new TarArchiveInputStream(new GzipCompressorInputStream(new FileInputStream(sourceFile)))) {
            // decompressing *.tar.gz files to tar
            TarArchiveEntry entry;
            while ((entry = fin.getNextTarEntry()) != null) {
                if (entry.isDirectory()) {
                    continue;
                }

                String filename = FilenameUtils.getName(entry.getName());
                String fileSuffix = Files.getFileExtension(filename);
                if (!suffix.contains(fileSuffix)) {
                    continue;
                }

                String tmpFilename = GeoIPConfig.getFilenameUpgrade(filename);
                File currentFile = new File(targetFolder, tmpFilename);
                if (!currentFile.exists()) {
                    currentFile.createNewFile();
                }
                logger.info("Uncompress file[{}] ", tmpFilename);
                // 将文件写出到解压的目录
                FileOutputStream fout = new FileOutputStream(currentFile);
                IOUtils.copy(fin, fout);
                fout.close();
                //copy 临时文件到正式文件
                //java.nio.file.Files.copy(Paths.get(targetPath, tmpFilename), Paths.get(targetPath, filename), StandardCopyOption.REPLACE_EXISTING);
            }

            logger.info("Finish uncompress file[{}] to[{}]", sourcePath, targetPath);
        } catch (IOException e) {
            logger.error("uncompressTarGzip IOException file[{}]", sourcePath, e);
        }
    }

    public static void move(String from, String to) {
        try {
//            java.nio.file.Files.copy(Paths.get(from), Paths.get(to), StandardCopyOption.REPLACE_EXISTING);
            File fromFile = new File(from);
            File toFile = new File(to);
            FileUtils.copyFile(fromFile, toFile, true);
            if (fromFile.exists()) {
                fromFile.delete();
            }
            logger.info("Move success.from[{}] to[{}]", from, to);
        } catch (IOException e) {
            logger.info("Move failed.from[{}] to[{}]", from, to);
        }
    }

    /**
     * @param type
     * @return
     */
    public static boolean isNeedUpdate(EGeoIPDBType type) {
        String filename = GeoIPConfig.dbFilename(type);
        File file = FileUtils.getFile(filename);
        if (!file.exists()) {
            logger.info("File [{}] not exist,need update", filename);
            return true;
        }

        long duration = GeoIPConfig.updateDuration();
        long lastModify = file.lastModified();
        if (System.currentTimeMillis() - lastModify > TimeUnit.MINUTES.toMillis(duration)) {
            logger.info("File [{}] need update,last modify time[{}]", filename, DateFormat.getDateTimeInstance(0, 0).format(new Date(lastModify)));
            return true;
        }

        return false;
    }

    public static boolean mkdirsIfNotExist(String path) {
        if (Strings.isNullOrEmpty(path)) {
            return false;
        }

        try {
            File rootPath = new File(path);
            if (!rootPath.exists()) {
                rootPath.mkdirs();
            }
            return true;
        } catch (Exception e) {
            logger.error("mkdir[{}] exception", path, e);
            return false;
        }
    }
}
