package common.log;

import com.google.common.base.Strings;
import org.apache.logging.log4j.core.config.Configurator;
import org.slf4j.*;


import java.io.File;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Map;
import java.util.Objects;


/**
 * @author
 */
public final class Logging {
    private final static Marker MARKER_STAT = MarkerFactory.getMarker("STAT");
    private final static Marker MARKER_ALARM = MarkerFactory.getMarker("ALARM");

    private static String logName = "";

    /**
     * set log name value
     *
     * @param name
     */
    public static Logging setLogName(String name) {
        if (!Strings.isNullOrEmpty(name)) {
            logName = name;
        }

        //此处返回null,目的便于链式调用？？
        return null;
    }

    /**
     * Set log property
     *
     * @param key
     * @param value
     */
    public static void setLoggerProperty(String key, String value) {
        if (Strings.isNullOrEmpty(key) || Strings.isNullOrEmpty(value)) {
            return;
        }

        System.setProperty(key, value);
        System.out.println(String.format("*** Set log property key=[%s] value=[%s]", key, value));
    }

    /**
     * Set log properties
     *
     * @param properties
     */
    public static void setLoggerProperties(Map<String, String> properties) {
        /**
         * support key:
         *     LOG_PATH:Log file path
         *     LOG_MODULE:Log Module name
         *     LOG_FILE_SIZE:Log file MAX size
         *     INFO_FILE_COUNT:Number of info files
         *     ERROR_FILE_COUNT:Number of error files
         *     STAT_FILE_COUNT:Number of stat files
         *     ALARM_FILE_COUNT:Number of alarm files
         *     LOG_AUTO_LOG_NAME: auto get log name 0:close 1:open
         */
        for (Map.Entry<String, String> entry : properties.entrySet()) {
            if (ELoggingProperty.isValid(entry.getKey())) {
                setLoggerProperty(entry.getKey(), entry.getValue());
            } else {
                System.out.println(String.format("*** Invalid property key=[%s] value=[%s]", entry.getKey(), entry.getValue()));
            }
        }
    }

    /**
     * Initialize log config
     *
     * @param fileName    :config file name
     * @param moduleName: log module name
     */
    public static void initialize(String moduleName, String fileName) {
        if (!Strings.isNullOrEmpty(moduleName)) {
            setLoggerProperty("LOG_MODULE", moduleName);
        }

        if (Strings.isNullOrEmpty(fileName)) {
            fileName = "config/log4j2.xml";
        }

        if (Paths.get(fileName).toFile().isFile()) {
            Configurator.initialize(null, fileName);
            System.out.println(String.format("Load log config file:[%s]", fileName));
        } else {
            System.out.println(String.format("Config file:[%s] not exist,use default", fileName));
        }
    }

    private static String getLogName() {
        if (!Strings.isNullOrEmpty(logName)) {
            return logName;
        }

        //关闭自动获取
        String autoGetName = System.getProperty("LOG_AUTO_GET_NAME");
        if (!Strings.isNullOrEmpty(autoGetName) && "1".contentEquals(autoGetName)) {
            // 通过调用栈获取调用方法名称(可能有性能问题)
            StackTraceElement caller = LoggingTools.findCaller();
            if (Objects.nonNull(caller)) {
                return String.format("%s.%s", caller.getClassName(), caller.getMethodName() + ":" + caller.getLineNumber());
            }
        }

        return Logging.class.getName();
    }

    /**
     * 获取日志对象
     *
     * @return
     */
    public static Logger getLogger() {
        return LoggerFactory.getLogger(getLogName());
    }

    /**
     * 获取日志对象
     *
     * @param name
     * @return
     */
    public static Logger getLogger(String name) {
        return LoggerFactory.getLogger(name);
    }

    /**
     * 获取Logger实例
     *
     * @param clazz
     * @return
     */
    public static Logger getLogger(Class clazz) {
        return LoggerFactory.getLogger(clazz.getName());
    }

    public void info(String msg) {
        getLogger().info(msg);
    }

    public void info(String format, Object... params) {
        getLogger().info(format, params);
    }

    public void info(String msg, Throwable t) {
        getLogger().info(msg, t);
    }

    /**
     * info 日志
     *
     * @param sessionId
     * @param clientIp
     * @param accountId
     * @param accountName
     * @param params
     */
    public static void info(String sessionId, String clientIp, String accountId, String accountName, String logSummary, String... params) {
        if (!getLogger().isInfoEnabled()) {
            return;
        }

        /**
         * 这个地方会导致filter中设置的mdc被清除
         */

        MDCTools.setMDC(sessionId, clientIp, accountId, accountName, logSummary);
        getLogger().info(LoggingMsgBuilder.formatParams(params));
        MDCTools.cleanMDC();
    }


    /**
     * info 日志
     *
     * @param params
     * @Param logSummary
     */
    public static void info2(String logSummary, String... params) {
        if (!getLogger().isInfoEnabled()) {
            return;
        }

        MDC.put(MDCConstants.MDC_LOG_SUMMARY, logSummary);
        getLogger().info(LoggingMsgBuilder.formatParams(params));
        MDC.remove(MDCConstants.MDC_LOG_SUMMARY);
    }


    /**
     * @param t
     * @param sessionId
     * @param clientIp
     * @param accountId
     * @param accountName
     * @param params
     */
    public static void info(Throwable t, String sessionId, String clientIp, String accountId, String accountName, String logSummary, String... params) {
        if (!getLogger().isInfoEnabled()) {
            return;
        }

        MDCTools.setMDC(sessionId, clientIp, accountId, accountName, logSummary);
        getLogger().info(LoggingMsgBuilder.formatParams(params), t);
        MDCTools.cleanMDC();
    }

    public void warn(String msg) {
        getLogger().info(msg);
    }

    public void warn(String format, Object... params) {
        getLogger().info(format, params);
    }

    public void warn(String msg, Throwable t) {
        getLogger().info(msg, t);
    }

    public static void wran2(String logSummary, String... params) {
        if (!getLogger().isInfoEnabled()) {
            return;
        }

        MDC.put(MDCConstants.MDC_LOG_SUMMARY, logSummary);
        getLogger().warn(LoggingMsgBuilder.formatParams(params));
        MDC.remove(MDCConstants.MDC_LOG_SUMMARY);
    }

    /**
     * Record warning logs
     *
     * @param sessionId
     * @param clientIp
     * @param accountId
     * @param accountName
     * @param params
     */
    public static void warn(String sessionId, String clientIp, String accountId, String accountName, String logSummary, String... params) {
        if (!getLogger().isWarnEnabled()) {
            return;
        }

        MDCTools.setMDC(sessionId, clientIp, accountId, accountName, logSummary);
        getLogger().warn(LoggingMsgBuilder.formatParams(params));
        MDCTools.cleanMDC();
    }

    /**
     * Record warning logs
     *
     * @param e
     * @param sessionId
     * @param clientIp
     * @param accountId
     * @param accountName
     * @param params
     */
    public static void warn(Throwable e, String sessionId, String clientIp, String accountId, String accountName, String logSummary, String... params) {
        if (!getLogger().isWarnEnabled()) {
            return;
        }

        MDCTools.setMDC(sessionId, clientIp, accountId, accountName, logSummary);
        getLogger().warn(LoggingMsgBuilder.formatParams(params), e);
        MDCTools.cleanMDC();
    }

    public void error(String msg) {
        getLogger().info(msg);
    }

    public void error(String format, Object... params) {
        getLogger().info(format, params);
    }

    public void error(String msg, Throwable t) {
        getLogger().info(msg, t);
    }

    /**
     * Record error logs
     *
     * @param sessionId
     * @param clientIp
     * @param accountId
     * @param accountName
     * @param params
     */
    public static void error(String sessionId, String clientIp, String accountId, String accountName, String logSummary, String... params) {
        if (!getLogger().isErrorEnabled()) {
            return;
        }

        MDCTools.setMDC(sessionId, clientIp, accountId, accountName, logSummary);
        getLogger().error(LoggingMsgBuilder.formatParams(params));
        MDCTools.cleanMDC();
    }

    /**
     * Record error logs
     *
     * @param e
     * @param sessionId
     * @param clientIp
     * @param accountId
     * @param accountName
     * @param params
     */
    public static void error(Throwable e, String sessionId, String clientIp, String accountId, String accountName, String logSummary, String... params) {
        if (!getLogger().isErrorEnabled()) {
            return;
        }

        MDCTools.setMDC(sessionId, clientIp, accountId, accountName, logSummary);
        getLogger().error(LoggingMsgBuilder.formatParams(params), e);
        MDCTools.cleanMDC();
    }

    public static void error2(String logSummary, String... params) {
        if (!getLogger().isInfoEnabled()) {
            return;
        }

        MDC.put(MDCConstants.MDC_LOG_SUMMARY, logSummary);
        getLogger().warn(LoggingMsgBuilder.formatParams(params));
        MDC.remove(MDCConstants.MDC_LOG_SUMMARY);
    }

    /**
     * Record stat msg
     *
     * @param msg
     */
    public static void stat(String msg) {
        getLogger().info(MARKER_STAT, LoggingMsgBuilder.buildStatMsg(msg));
    }

    /**
     * Record stat message
     *
     * @param key
     * @param value
     */
    public static void stat(String key, String value) {
        stat(LoggingMsgBuilder.buildStatMsg(key, value));
    }

    /**
     * Record stat message
     *
     * @param key
     * @param value
     * @param time
     * @param params
     */
    public static void stat(String key, String value, long time, String... params) {
        stat(LoggingMsgBuilder.buildStatMsg(key, value, time, params));
    }

    /**
     * Record alarm message
     *
     * @param sessionId
     * @param clientIp
     * @param accountId
     * @param accountName
     * @param alarmId
     * @param logSummary
     * @param params
     */
    public static void alarm(String sessionId, String clientIp, String accountId, String accountName, String alarmId, String logSummary, String... params) {
        if (!Logging.getLogger().isErrorEnabled()) {
            return;
        }

        String msg = LoggingMsgBuilder.formatParams(params);

        MDCTools.setMDC(sessionId, clientIp, accountId, accountName, logSummary);
        MDC.put(MDCConstants.MDC_ALARM_ID, alarmId);
        Logging.getLogger().info(MARKER_ALARM, msg);
        MDC.remove(MDCConstants.MDC_ALARM_ID);
        MDCTools.cleanMDC();
    }


}
