package common.log.filter;

import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;


import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import java.util.Objects;

public class HttpTools {

    /**
     * IP address original source
     */
    private static ImmutableList<String> ipHeaderNameList = ImmutableList.<String>builder()
            .add("x-forwarded-for")
            .add("CF-Connecting-IP")
            .add("Proxy-Client-IP")
            .add("WL-Proxy-Client-IP")

            .add("X-Forwarded-For")
            .add("HTTP_X_REAL_IP")
            .add("HTTP_CLIENT_IP")
            .add("HTTP_X_FORWARDED_FOR")
            .build();


    private static boolean isInvalidClientIp(String clientIp) {
        if (Strings.isNullOrEmpty(clientIp)) {
            return false;
        }

        if ("unknown".equalsIgnoreCase(clientIp)) {
            return false;
        }

        return true;
    }

    public static String getClientIp(HttpServletRequest request) {
        /*
           Get real client ip,support CloudFlare,Aliyun,AWS proxy
         */
        String remoteHost = null;
        for (String headerName : ipHeaderNameList) {
            remoteHost = request.getHeader(headerName);
            if (isInvalidClientIp(remoteHost)) {
                break;
            }
        }

        //CAN NOT get from header,get from request
        if (!isInvalidClientIp(remoteHost)) {
            remoteHost = request.getRemoteAddr();
        }

        //对于通过多个代理的情况，第一个IP为客户端真实IP,多个IP按照','分割
        if (!Strings.isNullOrEmpty(remoteHost)) {
            int pos = remoteHost.indexOf(",");
            if (pos > 0) {
                remoteHost = remoteHost.substring(0, pos);
            }
        }

        return Strings.nullToEmpty(remoteHost);
    }


    public static String getParameter(HttpServletRequest request, String key) {
        if (request == null) {
            return "";
        }

        if (Strings.isNullOrEmpty(key)) {
            return "";
        }

        return request.getParameter(key);
    }

    public static String getCookie(HttpServletRequest request, String key) {
        if (request == null) {
            return "";
        }

        if (Strings.isNullOrEmpty(key)) {
            return "";
        }

        /**
         * 这里每次获取都要遍历整个cookie
         */
        Cookie[] cookies = request.getCookies();
        if (Objects.nonNull(cookies)){
            for (Cookie cookie : cookies) {
                if (key.contentEquals(cookie.getName())) {
                    return cookie.getValue();
                }
            }
        }

        return "";
    }

    public static String getHeader(HttpServletRequest request, String key) {
        if (request == null) {
            return "";
        }

        if (Strings.isNullOrEmpty(key)) {
            return "";
        }
        return Strings.nullToEmpty(request.getHeader(key));
    }

    /**
     * Get arg from http header:Cookie-> Http Parameter -> Http Header
     *
     * @param request
     * @param key
     * @return
     */
    public static String getHttpParameter(HttpServletRequest request, String key) {
        String value = getCookie(request, key);
        if (!Strings.isNullOrEmpty(value)) {
            return value;
        }

        value = getParameter(request, key);
        if (!Strings.isNullOrEmpty(value)) {
            return value;
        }

        value = getHeader(request, key);
        if (!Strings.isNullOrEmpty(value)) {
            return value;
        }

        return "";
    }


}
