package common.log.filter;


import java.io.IOException;
import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;


import com.google.common.base.Strings;
import common.log.MDCConstants;
import org.slf4j.MDC;

/**
 * @author Frank.Huang
 */
public class MDCServletFilter implements Filter {

    @Override
    public void init(FilterConfig filterConfig) throws ServletException {
        //do nothing
    }

    @Override
    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain) throws IOException, ServletException {

        insertIntoMDC(request);
        try {
            chain.doFilter(request, response);
        } finally {
            clearMDC();
        }
    }

    @Override
    public void destroy() {
        //Do nothing
    }


    /**
     * insert in MDC
     *
     * @param request
     */
    private void insertIntoMDC(ServletRequest request) {
        MDC.put(MDCConstants.MDC_SESSION_ID, getSessionId((HttpServletRequest) request));
        MDC.put(MDCConstants.MDC_CLIENT_IP, HttpTools.getClientIp((HttpServletRequest) request));
        MDC.put(MDCConstants.MDC_ACCOUNT_ID, getAccountId((HttpServletRequest) request));
        MDC.put(MDCConstants.MDC_ACCOUNT_NAME, getAccountName((HttpServletRequest) request));
    }

    /**
     * clean MDC value
     */
    private void clearMDC() {
        MDC.remove(MDCConstants.MDC_SESSION_ID);
        MDC.remove(MDCConstants.MDC_CLIENT_IP);
        MDC.remove(MDCConstants.MDC_ACCOUNT_ID);
        MDC.remove(MDCConstants.MDC_ACCOUNT_NAME);
    }


    private String getSessionId(HttpServletRequest request) {
        return HttpTools.getHttpParameter(request, "sessionId");
    }

    private String getAccountId(HttpServletRequest request) {
        /**
         * accountId->uid
         */
        String value = HttpTools.getHttpParameter(request, "accountId");
        if (!Strings.isNullOrEmpty(value)) {
            return value;
        }

        value = HttpTools.getHttpParameter(request, "uid");
        if (!Strings.isNullOrEmpty(value)) {
            return value;
        }
        return "";
    }

    private String getAccountName(HttpServletRequest request) {
        /**
         * accountName -> mac -> did
         */
        String value = HttpTools.getHttpParameter(request, "accountName");
        if (!Strings.isNullOrEmpty(value)) {
            return value;
        }

        value = HttpTools.getHttpParameter(request, "mac");
        if (!Strings.isNullOrEmpty(value)) {
            return value;
        }

        value = HttpTools.getHttpParameter(request, "did");
        if (!Strings.isNullOrEmpty(value)) {
            return value;
        }

        return "";
    }


}
