package common.stcode;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.management.ManagementFactory;
import java.lang.management.RuntimeMXBean;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.UUID;

import com.google.common.base.Strings;

/**
 * snowflake的结构如下(每部分用-分开):
 * <p>
 * 1位标识部分     -         10位节点部分  -   12位序列号部分 -      41位时间戳部分
 * 0 - - 00000 - 00000 - 000000000000 -0000000000 0000000000 0000000000 0000000000 0
 * 第一位为未使用，接下来的41位为毫秒级时间(41位的长度可以使用69年)，然后是5位datacenterId和5位workerId(10位的长度最多支持部署1024个节点） ，最后12位是毫秒内的计数（12位的计数顺序号支持每个节点每毫秒产生4096个ID序号）
 * <p>
 * 一共加起来刚好64位，为一个Long型。(转换成字符串长度为18)
 * <p>
 * snowflake生成的ID整体上按照时间自增排序，并且整个分布式系统内不会产生ID碰撞（由machine id区分），并且效率较高。据说：snowflake每秒能够产生26万个ID。
 * <p>
 * <p>
 * https://github.com/twitter-archive/snowflake
 * https://github.com/souyunku/SnowFlake
 * <p>
 * Usage :
 * 1.Get instance
 * common.stcode.SnowFlakeID uniqueIdGenerator = common.stcode.SnowFlakeID.getInstance(0);
 * 2.Call method to generator new id
 * uniqueIdGenerator.nextId()
 * <p>
 * Example：
 * common.stcode.SnowFlakeID uniqueIdGenerator = common.stcode.SnowFlakeID.getInstance(0);
 * <p>
 * long start = System.currentTimeMillis();
 * for (int i = 0; i < 100L; i++) {
 * System.out.println(uniqueIdGenerator.nextId());
 * }
 * System.out.println(System.currentTimeMillis() - start);
 */
public class SnowFlakeID {
    private static final Logger logger = LoggerFactory.getLogger(SnowFlakeID.class);
    /**
     * 起始的时间戳：2023-06-01 00:00:00
     */
    private final static long START_TIMESTAMP = 1685548800000L;

    /**
     * 每一部分占用的位数
     */
    //机器标识占用的位数(1~10)
    private final static long MACHINE_BIT = 10;
    //序列号占用的位数(11~23)
    private final static long SEQUENCE_BIT = 12;
    //时间戳占用位(23~64)
    private final static long TIMESTAMP_BIT = 41;

    /**
     * 每一部分的最大值
     */

    private final static long MAX_MACHINE_NUM = ~(-1L << MACHINE_BIT);
    private final static long MAX_SEQUENCE = ~(-1L << SEQUENCE_BIT);
    private final static long MAX_TIMESTAMP = ~(-1L << TIMESTAMP_BIT);

    /**
     * 步长, 1024
     */
    private final static long SEQUENCE_STEP_SIZE = 1024L;
    /**
     * 基础序列号, 每发生一次时钟回拨, basicSequence += stepSize
     */
    private long basicSequence = 0L;

    private static String hostname = "";

    /**
     * 每一部分向左的位移
     */
    private final static long MACHINE_LEFT = SEQUENCE_BIT + TIMESTAMP_BIT;
    private final static long SEQUENCE_LEFT = TIMESTAMP_BIT;
    private final static long TIMESTMP_LEFT = 0;


    private long machineId = 1L;
    //序列号
    private long sequence = 0L;
    //上一次时间戳
    private long lastTimestamp = -1L;

    private static class UniqueIdGeneratorInstance {
        private final static SnowFlakeID instance = new SnowFlakeID();
    }

    public static SnowFlakeID getInstance(long machineId) {
        UniqueIdGeneratorInstance.instance.setMachineId(machineId);
        return UniqueIdGeneratorInstance.instance;
    }

    private SnowFlakeID() {
    }


    /**
     * Set machine id
     *
     * @param machineId:机器ID
     * @return : machineId
     */
    private long setMachineId(long machineId) {
        if (machineId > 0 && machineId <= MAX_MACHINE_NUM) {
            this.machineId = machineId;
            return this.machineId;
        }

        //Machine Id 不合法,取当前进程id
        String machineStr = getMachineStr();
        this.machineId = Math.abs(machineStr.hashCode() % (int) MAX_MACHINE_NUM);
        logger.info("common.stcode.SnowFlakeID:MachineId[{}] MachineStr[{}]", this.machineId, machineStr);
        return this.machineId;
    }


    private String getMachineStr() {
        int pid = getCurrentPid();
        if (pid == -1) {
            pid = UUID.randomUUID().hashCode();
        }
        return getHostname() + "@" + pid;
    }

    /**
     * 产生下一个ID
     *
     * @return
     */
    public synchronized long nextId() {
        long currentTimestamp = getNewTimestamp();
        if (currentTimestamp < lastTimestamp) {
            /**
             * 时钟回拨:起始步长增加
             *   限制：每毫秒生成的不会超过1024
             */
            logger.warn("SnowFlakeID:Clock moved backwards.currentTimestamp:[{}],lastStamp[{}],Offset[{}]", currentTimestamp, lastTimestamp, currentTimestamp - lastTimestamp);
            basicSequence += SEQUENCE_STEP_SIZE;
            if (basicSequence > MAX_SEQUENCE) {
                basicSequence = 0;
                currentTimestamp = getNextMill();
            }
        }

        if (currentTimestamp == lastTimestamp) {
            //相同毫秒内，序列号自增
            sequence = (sequence + 1) & MAX_SEQUENCE;
            //同一毫秒的序列数已经达到最大
            if (sequence == 0L) {
                currentTimestamp = getNextMill();
            }
        } else {
            //不同毫秒内，序列号置为0
            sequence = basicSequence;
        }

        lastTimestamp = currentTimestamp;

        return machineId << MACHINE_LEFT                 //机器标识部分
                | sequence << SEQUENCE_LEFT              //序列号部分
                | (currentTimestamp - START_TIMESTAMP);  //时间戳部分
    }

    public static long parseMachineId(long code) {
        return (code >> MACHINE_LEFT) & MAX_MACHINE_NUM;
    }

    public static long parseSequence(long code) {
        return (code >> SEQUENCE_LEFT) & MAX_SEQUENCE;
    }

    public static long parseTimestamp(long code) {
        return code & MAX_TIMESTAMP;
    }

    public static long parseAbsTimestamp(long code) {
        return parseTimestamp(code) + START_TIMESTAMP;
    }

    private long getNextMill() {
        long mill = getNewTimestamp();
        while (mill <= lastTimestamp) {
            mill = getNewTimestamp();
        }
        return mill;
    }

    private long getNewTimestamp() {
        return System.currentTimeMillis();
    }

    public static final int getCurrentPid() {
        RuntimeMXBean runtimeMXBean = ManagementFactory.getRuntimeMXBean();
        String processName = runtimeMXBean.getName();
        if (processName.indexOf('@') != -1) {
            String pid = processName.substring(0, processName.indexOf('@'));
            if (!Strings.isNullOrEmpty(pid)) {
                return Integer.valueOf(pid);
            }
        }

        return -1;
    }

    private static String getHostname() {
        if (!Strings.isNullOrEmpty(hostname)) {
            return hostname;
        }

        try {
            hostname = InetAddress.getLocalHost().getHostName();
            if (!Strings.isNullOrEmpty(hostname)) {
                return hostname;
            }
        } catch (UnknownHostException e) {
            // failed;  try alternate means.
        }

        // try environment properties.
        hostname = System.getenv("COMPUTERNAME");
        if (!Strings.isNullOrEmpty(hostname)) {
            return hostname;
        }

        hostname = System.getenv("HOSTNAME");
        if (!Strings.isNullOrEmpty(hostname)) {
            return hostname;
        }

        // undetermined.
        return null;
    }
}
