package common.stcode;

import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;

import java.util.List;

public class StCodeValidate {
    //版本号
    public static final List<Long> supportVersions = ImmutableList.of(3L);
    public static final long MIN_FACTORY_ID = 1; //最小工厂ID
    public static final long MAX_FACTORY_ID = 9999; //最大工厂ID
    public static final long MAX_CHECK_CODE = ~(-1L << 5); //校验码最大长度

    private static final NumberStringifier r62 = new NumberStringifier("0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz".toCharArray());

    /**
     * 校验ST Code是否正确
     *
     * @param stCode:st code
     * @return 0:成功 <0 失败
     * -1:st 为空
     * -2:非法字符
     * -3:最小长度异常
     * -4:不支持的版本
     * -5:版本长度异常
     * -6:非法的工厂ID
     * -7:校验码错误
     */
    public static int validate(String stCode) {
        if (Strings.isNullOrEmpty(stCode)) {
            return -1;
        }
        String pattern = "^[0-9A-Za-z]+$";
        if ( !stCode.matches(pattern) ) {
            return -2;
        }

        //check len
        int len = stCode.length();
        if (len <= 6) {
            return -3;
        }

        String version = stCode.substring(4, 6);
        long ver;
        try {
            ver = Long.valueOf(version, 16);
        } catch ( NumberFormatException ex ) {
            return -4;
        }
        if (!supportVersions.contains(ver)) {
            return -4;
        }
        //v3版本为3 长度为24
        if (ver == 3 && len != 20) {
            return -5;
        }

        String factoryId = stCode.substring(0, 4);
        if (!validateFactoryId(factoryId, MIN_FACTORY_ID, MAX_FACTORY_ID)) {
            return -6;
        }

        String checksumStrFromCode = stCode.substring(6, 8);
        String codeStr = stCode.substring(8);
        long checksumFromCode = r62.parse(checksumStrFromCode);
        long checksum = calcChecksum(r62.parse(codeStr), MAX_CHECK_CODE);
        if (checksum != checksumFromCode) {
            return -7;
        }

        return 0;
    }

    protected static String toR62String(long num, int maxLength) {
        return Strings.padStart(r62.stringify(num), maxLength, '0');
    }


    private static boolean validateFactoryId(String factoryId, long min, long max) {
        int id = Integer.parseInt(factoryId, 16);
        return id >= min && id <= max;
    }

    /**
     * 计算校验码
     *
     * @param code 原始数字
     * @return 校验码
     */
    public static long calcChecksum(long code, long maxCheckCode) {
        String strOriginId = String.valueOf(code);
        int[] numbers = new int[strOriginId.length()];
        for (int i = 0, length = strOriginId.length(); i < length; i++) {
            numbers[i] = Character.getNumericValue(strOriginId.charAt(i));
        }
        for (int i = numbers.length - 2; i >= 0; i -= 2) {
            numbers[i] <<= 1;
            numbers[i] = numbers[i] / 10 + numbers[i] % 10;
        }

        int validationCode = 0;
        for (int number : numbers) {
            validationCode += number;
        }
        validationCode *= 9;

        return validationCode % maxCheckCode;
    }
}
