package common.web.tools.filter.common;

import com.google.common.base.Charsets;
import com.google.common.base.Strings;
import com.google.gson.Gson;
import common.base.tools.type.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpMethod;

import javax.servlet.ServletInputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;

/**
 * RequestWrapper for filter
 */
public class WebRequestWrapper extends HttpServletRequestWrapper {
    private static final Logger logger = LoggerFactory.getLogger(WebRequestWrapper.class);
    private Map<String, String[]> parameterMap = new HashMap<String, String[]>();
    private String bodyStr = "";
    private String originalBodyStr = "";
    private Gson gson = new Gson();

    public WebRequestWrapper(HttpServletRequest request) {
        super(request);
        if (HttpMethod.POST.name().equalsIgnoreCase(request.getMethod())) {
            bodyStr = getRequestBody(request);
        }

        parameterMap.putAll(request.getParameterMap());
    }

    // 重写getParameter，代表参数从当前类中的map获取
    @Override
    public String getParameter(String name) {
        String[] values = parameterMap.get(name);
        if (values == null || values.length == 0) {
            return null;
        }
        return values[0];
    }

    // getParameterValues，代表参数从当前类中的map获取
    @Override
    public String[] getParameterValues(String name) {
        return parameterMap.get(name);
    }

    @Override
    public Map<String, String[]> getParameterMap() {
        return parameterMap;
    }

    @Override
    public Enumeration getParameterNames() {
        return Collections.enumeration(this.parameterMap.keySet());
    }

    @Override
    public BufferedReader getReader() throws IOException {
        return new BufferedReader(new InputStreamReader(getInputStream(), getCharacterEncoding()));
    }

    @Override
    public ServletInputStream getInputStream() throws IOException {
        return new ServletInputStreamWrapper((Strings.nullToEmpty(originalBodyStr)).getBytes(Charsets.UTF_8));

        //        final ByteArrayInputStream byteArrayInputStream = new ByteArrayInputStream((Strings.nullToEmpty(requestBodyStr)).getBytes(Charsets.UTF_8));
//        return new DelegatingServletInputStream(byteArrayInputStream);


//        return new ServletInputStream() {
//            @Override
//            public boolean isFinished() {
//                return false;
//            }
//
//            @Override
//            public boolean isReady() {
//                return true;
//            }
//
//            @Override
//            public void setReadListener(ReadListener readListener) {
//
//            }
//
//            @Override
//            public int read() throws IOException {
//                return byteArrayInputStream.read();
//            }
//        };
    }


    public void addAllParameters(Map<String, Object> otherParams) {// 增加多个参数
        if (!CollectionUtils.isNullOrEmpty(otherParams)) {
            for (Map.Entry<String, Object> entry : otherParams.entrySet()) {
                addParameter(entry.getKey(), entry.getValue());
            }
        }
    }

    public void setOriginalBodyStr(String originalBodyStr){
        this.originalBodyStr = originalBodyStr;
    }

    public void addParameter(String name, Object value) {// 增加参数
        if (value != null) {
            if (value instanceof String[]) {
                parameterMap.put(name, (String[]) value);
            } else if (value instanceof String) {
                parameterMap.put(name, new String[]{(String) value});
            } else {
                parameterMap.put(name, new String[]{String.valueOf(value)});
            }
        }
    }

    private String getRequestBody(HttpServletRequest request) {
        String requestBody = getRequestBodyString(request);
        if (!Strings.isNullOrEmpty(requestBody)) {
            Map<String, Object> postMap = gson.fromJson(requestBody, Map.class);
            if (!CollectionUtils.isNullOrEmpty(postMap)) {
                postMap.forEach((k, v) -> {
                    addParameter(k, v);
                });
            }
        }

        return Strings.nullToEmpty(requestBody);
    }


    public static String getRequestBodyString(HttpServletRequest request) {
        BufferedReader br = null;
        StringBuilder sb = new StringBuilder("");
        try {
            br = request.getReader();
            String str;
            while ((str = br.readLine()) != null) {
                sb.append(str);
            }
            br.close();
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            if (null != br) {
                try {
                    br.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        return sb.toString();
    }

}
