package common.web.tools.filter.crypt.filter;


import com.google.common.base.Strings;
import com.google.gson.Gson;
import common.base.tools.statistics.ApiQpsStatisticsTools;
import common.base.tools.sys.SystemTools;
import common.config.tools.config.ConfigTools3;
import common.web.tools.filter.FilterConstant;
import common.web.tools.filter.common.WebResponseWrapper;
import common.web.tools.filter.crypt.CryptACL;
import common.web.tools.filter.crypt.CryptMsgTools;
import common.web.tools.filter.crypt.ResponseCacheTool;
import common.web.tools.filter.crypt.model.ApiCryptMsg;
import common.web.tools.filter.crypt.model.ApiCryptMsg4Live;
import common.web.tools.http.HttpTools;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.filter.OncePerRequestFilter;

import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;


public class ResponseCryptFilter extends OncePerRequestFilter {
    private static final Logger logger = LoggerFactory.getLogger(ResponseCryptFilter.class);
    private Gson gson = new Gson();

    @Override
    protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response,
                                    FilterChain filterChain) throws ServletException, IOException {
//        RequestCryptWrapper reqWrapper = new RequestCryptWrapper(request);
        Long requestArrivalTime = null;
        Map responseMap = new ConcurrentHashMap();
        Object obj = request.getAttribute(FilterConstant.SERVER_PARAM_REQUEST_ARRIVAL_TIME);
        try {
            requestArrivalTime = (Long) obj;
        } catch (Throwable ignored) {
        }

        try {
            String v = request.getParameter("v");
            if (Strings.isNullOrEmpty(v)) {
                v = request.getParameter("prot");
            }

            boolean isGetServerTime =
                    BooleanUtils.toBoolean(request.getHeader(FilterConstant.CLIENT_PARAM_GET_SERVER_TIME));
            if (isGetServerTime) {
                response.addHeader(FilterConstant.SERVER_PARAM_SERVER_TIME, Long.toString(System.currentTimeMillis()));
            }

            long clientTime = NumberUtils.toLong(request.getHeader(FilterConstant.CLIENT_PARAM_CLIENT_TIME), 0L);
            if (clientTime > 0L) {
                response.addHeader(FilterConstant.CLIENT_PARAM_CLIENT_TIME, Long.toString(clientTime));
            }

            String url = HttpTools.getRequestURI(request);

            if (Strings.isNullOrEmpty(v) || CryptACL.isResponseCryptDisable(url)) {
                filterChain.doFilter(request, response);
            } else {
                OutputStream out = response.getOutputStream();
                WebResponseWrapper wrapper = new WebResponseWrapper(response);
                filterChain.doFilter(request, wrapper);

                if (CryptACL.isContentTypeCrypt(wrapper.getContentType())) {
                    responseMap = gson.fromJson(new String(wrapper.getData()), Map.class);
                    out.write(wrapper.getData());
                } else {
                    byte[] data = getData(wrapper, request, response, isGetServerTime, url, v);
                    if (Objects.nonNull(data)) {
                        responseMap = gson.fromJson(new String(data), Map.class);
                    }
                    String cryptMsg = encryptMsg(url, v, data);
                    data = cryptMsg.getBytes("utf-8");
                    out.write(data);
                }
            }
        } finally {
            if (requestArrivalTime != null && ConfigTools3.getBoolean("statistics.qps.report", false)) {
                String apiName = request.getRequestURI();
                String serverAddress = request.getLocalAddr();
                String serverPort = String.valueOf(request.getLocalPort());
                String httpResponseCode = String.valueOf(response.getStatus());
                int errCode = (Integer) responseMap.get("errCode");
                String errMsg = String.valueOf(responseMap.get("errMsg"));
                long duration = System.currentTimeMillis() - requestArrivalTime;
                ApiQpsStatisticsTools.statisticRequests(apiName, serverAddress, serverPort, httpResponseCode, errCode
                        , errMsg, duration);
            }
        }
    }

    @Override
    public void destroy() {
        // empty interface implementation
    }

    private byte[] getData(WebResponseWrapper wrapper,
                           HttpServletRequest request,
                           HttpServletResponse response,
                           boolean isGetServerTime, String url, String v)
            throws IOException {
        String key = getKey(request);
        String msg = getCache(request, key);
        byte[] data;

        if (isGetServerTime) {
            if (StringUtils.isNotEmpty(msg)) {
                data = msg.getBytes("utf-8");
                //logger.info("[ResponseCryptFilter] new api hit cache");
            } else {
                data = wrapper.getData();
/*                String cryptMsg = encryptMsg(url, v, data);
                data = cryptMsg.getBytes();*/
                if (StringUtils.isNotEmpty(key))
                    /*ResponseCacheTool.put(key, cryptMsg);*/
                    ResponseCacheTool.put(key, new String(data, "utf-8"));
                //logger.info("[ResponseCryptFilter] new api misses cache");
            }
        } else {
            if (StringUtils.isEmpty(msg)) {
                data = wrapper.getData();
                msg = new String(data, "utf-8");

                if (StringUtils.isNotEmpty(key))
                    ResponseCacheTool.put(key, msg);
                //logger.info("[ResponseCryptFilter] old api misses cache");
            } else {
                msg = updateServerTime(request, msg);
                data = msg.getBytes();
                //logger.info("[ResponseCryptFilter] old api hit cache");
            }

/*            String cryptMsg = encryptMsg(url, v, data);
            data = cryptMsg.getBytes("utf-8");*/
        }

        return data;
    }

    private String getKey(HttpServletRequest request) {
        Object obj = request.getAttribute(FilterConstant.SERVER_PARAM_CACHE_KEY);
        return obj == null ? "" : obj.toString();
    }

    private String getCache(HttpServletRequest request, String key) {
        Object obj = request.getAttribute(FilterConstant.SERVER_PARAM_USE_CACHE);
        boolean isUseCache = obj != null && Boolean.parseBoolean(obj.toString());

        String msg = "";
        if (isUseCache && !StringUtils.isEmpty(key)) {
            msg = StringUtils.trimToEmpty(ResponseCacheTool.get(key));
        }

        //logger.info("[ResponseCryptFilter] cacheKey[{}], isUseCache[{}]", key, isUseCache);

        return msg;
    }

    private String encryptMsg(String url, String v, byte[] data) throws IOException {
        String cryptMsg = "";
        if (CryptMsgTools.isLiveMsg(v)) {
            cryptMsg = getCryptMessage4Live(url, v, data);
        } else {
            cryptMsg = getCryptMessage(url, v, data);
        }

        return cryptMsg;
    }

    private String updateServerTime(HttpServletRequest request, String msg) {
        Object obj = request.getAttribute("serverTimeField");
        String serverTimeField = obj == null ? "" : obj.toString();
        if (StringUtils.isNotEmpty(serverTimeField)) {
            return updateServerTime(msg, serverTimeField);
        }

        return msg;
    }

    private String updateServerTime(String json, String filedName) {
        int pos = StringUtils.indexOf(json, filedName);
        if (pos < 0)
            return json;

        StringBuilder builder = new StringBuilder();

        while (json.charAt(pos) != ':') pos++;
        builder.append(json, 0, pos + 1);
        builder.append(System.currentTimeMillis());
        while (json.charAt(pos) != ',') pos++;
        builder.append(json, pos, json.length());

        return builder.toString();
    }

    private String getCryptMessage(String url, String protocol, byte[] data) throws IOException {
        boolean cryptEnable = CryptACL.isCryptEnable(url);
        try {
            ApiCryptMsg msg = new ApiCryptMsg();
            msg.setH(SystemTools.getHostnameMD5());
            msg.setT(System.currentTimeMillis());

            if (cryptEnable && CryptMsgTools.isSupportCrypt(protocol)) {
                msg.setV(protocol);
                msg.setM(CryptMsgTools.encryptMsg(protocol, data));
            } else {
                msg.setV("0");
                msg.setM(new String(data, "UTF-8"));
            }
            msg.setS(CryptMsgTools.sign(protocol, msg.getM()));

            return gson.toJson(msg);

        } finally {

        }
    }

    private String getCryptMessage4Live(String url, String prot, byte[] data) throws IOException {
        boolean cryptEnable = CryptACL.isCryptEnable(url);
        try {
            ApiCryptMsg4Live msg = new ApiCryptMsg4Live();

            if (cryptEnable && CryptMsgTools.isSupportCrypt(prot)) {
                msg.setProt(prot);
                msg.setPayload(CryptMsgTools.encryptMsg(prot, data));
            } else {
                msg.setProt("0");
                msg.setPayload(new String(data, "UTF-8"));
            }
            msg.setSignature(CryptMsgTools.sign(prot, msg.getPayload()));

            return gson.toJson(msg);

        } finally {

        }
    }


}
