package common.web.tools.statistics.scheduler;

import com.google.common.collect.ImmutableMap;
import common.base.tools.statistics.*;
import common.base.tools.sys.SystemTools;
import common.config.tools.config.ConfigTools3;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;

public class WebApiStatisticsReportTools {
    private static final Logger logger = LoggerFactory.getLogger(WebApiStatisticsReportTools.class);

    /**
     * Report slow request
     *
     * @param key
     * @param time
     */
    public static void reportSlowRequest(String key, long time) {
        long slowMin = ConfigTools3.getLong("statistics.slow", 200_000L);
        if (slowMin == 0 || time < slowMin) {
            return;
        }

        logger.info("[SLOW]name:[{}] time[{}]", key, time);
        String metricName = String.format("%s_time_slow", ConfigTools3.getString("statistics.metric.prefix",
                "common_statistics"));
        ImmutableMap tags = ImmutableMap.of("host", SystemTools.getHostname(),
                "item", key);
        ImmutableMap values = ImmutableMap.of("time", time);
        MetricTools2.postEvent(metricName, tags, values);
    }

    /**
     * report request time
     *
     * @param key
     * @param count
     */
    public static void reportCount(String key, long count) {
        String metricName = String.format("%s_count", ConfigTools3.getString("statistics.metric.prefix",
                "common_statistics"));
        ImmutableMap tags = ImmutableMap.of("host", SystemTools.getHostname(),
                "item", key);
        ImmutableMap values = ImmutableMap.of("count", count);
        MetricTools2.postEvent(metricName, tags, values);
    }

    public static void reportTimeConsumed(String key, long count, long avg) {
        String metricName = String.format("%s_time_consumed", ConfigTools3.getString("statistics.metric.prefix",
                "common_statistics"));
        ImmutableMap tags = ImmutableMap.of("host", SystemTools.getHostname(),
                "item", key);
        ImmutableMap values = ImmutableMap.of("count", count, "avg", avg);
        MetricTools2.postEvent(metricName, tags, values);
    }

    public static void reportTimeConsumedByStatus(String key, StatTimeItem timeItem) {
        int level = ConfigTools3.getInt(String.format("statistics.level.%s", key), 0);
        if (level <= 0) {
            return;
        }

        long count = timeItem.getCountGELevel(level);
        if (count == 0) {
            return;
        }

        String reportKey = String.format("%s.%s", timeItem.getKey(), key);

        String metricName = String.format("%s_time_status", ConfigTools3.getString("statistics.metric.prefix",
                "common_statistics"));
        ImmutableMap tags = ImmutableMap.of("host", SystemTools.getHostname(),
                "item", reportKey);
        ImmutableMap values = ImmutableMap.of("count", count);
        MetricTools2.postEvent(metricName, tags, values);
    }


    public static void reportQps(Map.Entry element) {
        String metricName = String.format("%s_api_qps", ConfigTools3.getString("statistics.metric.prefix",
                "common_statistics"));
        ServiceStatisticItem key = (ServiceStatisticItem) element.getKey();
        QpsItem value = (QpsItem) element.getValue();
        if (Objects.nonNull(key) && Objects.nonNull(value)) {
            logger.info("apiStatistics info : [{}]--[{}]--[{}]--[{}]", key.toString(), value.toString(),
                    (System.currentTimeMillis() - value.getArriveTime()) / 1000,
                    value.getVisitCount() / ((System.currentTimeMillis() - value.getArriveTime()) / 1000.00));

            Map<String, Object> filedMap = new ConcurrentHashMap();
            filedMap.put("apiName", key.getApiName());
            filedMap.put("serverAddress", key.getServerAddress());
            filedMap.put("httpCode", key.getHttpResponseCode());
            filedMap.put("errCode", key.getErrCode());
            filedMap.put("errMsg", key.getErrMsg());

            filedMap.put("visitCount", value.getVisitCount());
            filedMap.put("maxQps", (Math.round(1000.0D / value.getMinDurationMs() * 100)) / 100.0);
            filedMap.put("minQps", (Math.round(1000.0D / value.getMaxDurationMs() * 100)) / 100.0);
            filedMap.put("totalQps",
                    (Math.round(value.getVisitCount() / ((System.currentTimeMillis() - value.getArriveTime()) / 1000) * 100)) / 100.0);

            MetricTools2.postMercuryEvent(metricName, metricName, filedMap);
        }
    }

}
