package common.web.tools.diagnosis;

import com.google.common.collect.Maps;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.net.URL;
import java.util.Enumeration;
import java.util.Map;
import java.util.jar.JarFile;
import java.util.jar.Manifest;

public class ManifestTools {
    private static final Logger logger = LoggerFactory.getLogger(ManifestTools.class);
    private Map<String, Object> manifestMap = Maps.newHashMap();


    private static class ManifestInfoHolder {
        private static final ManifestTools INSTANCE = new ManifestTools();
    }

    public static ManifestTools getInstance() {
        return ManifestInfoHolder.INSTANCE;
    }

    private ManifestTools() {
        loadManifest();
    }

    private void loadManifest() {
        Enumeration resEnum;
        try {
            // get a list of all manifest files found in the jars loaded by the app
            resEnum = Thread.currentThread().getContextClassLoader().getResources(JarFile.MANIFEST_NAME);
            while (resEnum.hasMoreElements()) {
                try {
                    URL url = (URL) resEnum.nextElement();
                    String jarName = url.toString();

                    if (!jarName.contains("BOOT-INF")) {
                        //skip springboot depends libs
                        jarName = jarName.replace("jar:file:", "")
                            .replace("!/META-INF/MANIFEST.MF", "");

                        logger.info("load:{}", jarName);

                        JarFile jarFile = new JarFile(jarName, false);
                        Manifest manifest = jarFile.getManifest();

                        manifest.getMainAttributes().forEach((k1, v1) -> {
                            manifestMap.put(k1.toString(), v1);
                            logger.info("{}={}", k1.toString(), v1);
                        });
                    }
                } catch (Exception e) {
                    // Silently ignore wrong manifests on classpath?
                    logger.error("Exception", e);
                }
            }
        } catch (IOException e1) {
            // Silently ignore wrong manifests on classpath?
            logger.error("Exception", e1);
        }
    }

    public String getManifest(String name) {
        if (manifestMap.isEmpty()) {
            return "";
        }

        return manifestMap.getOrDefault(name, "").toString();
    }
}
