package common.web.tools.statistics.scheduler;

import com.google.common.base.Strings;
import common.base.tools.stat.StatTools;
import common.base.tools.statistics.*;
import common.base.tools.type.CollectionUtils;
import common.config.tools.config.ConfigTools3;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Service;

import java.util.Map;

import static common.base.tools.statistics.ApiQpsStatisticsTools.getAndResetVisit;
import static common.web.tools.statistics.scheduler.WebApiStatisticsReportTools.reportQps;


@Service
public class WebApiStatisticsScheduler {

    private static final Logger logger = LoggerFactory.getLogger(WebApiStatisticsScheduler.class);
    static long lastReportTS = 0;

    @Scheduled(fixedRate = 30000L)
    public void statisticsScheduler() {
        long cycle = ConfigTools3.getLong("api.statistics.cycle", 60 * 1000L);
        if (System.currentTimeMillis() - lastReportTS > cycle) {
            reportAndPrint();
            lastReportTS = System.currentTimeMillis();
        }
    }

    @Scheduled(fixedDelay = 60000)
    public void localStatisticsScheduler() {
        String msg = StatTools.dumpMutilline(false);
        if (!Strings.isNullOrEmpty(msg)) {
            logger.info("LOCAL-STAT:{}", msg);
        }
    }

    public void reportAndPrint() {
        if (ConfigTools3.getBoolean("statistics.out.log", true)) {
            boolean printPretty = ConfigTools3.getBoolean("statistics.out.pretty", true);
            String countStats = CountStatisticsTools2.dump(printPretty);
            String timeStats = TimeStatisticsTools2.dump(printPretty);

            if (!Strings.isNullOrEmpty(countStats) || !Strings.isNullOrEmpty(timeStats)) {
                logger.info("CountStats:{} TimeStats:{}", countStats, timeStats);
            }
        }

        if (ConfigTools3.getBoolean("statistics.report", true)) {
            reportCount2Metric();
            reportTime2Metric();
        }

        if (ConfigTools3.getBoolean("statistics.qps.report", true)) {
            reportQpiQps2Metric();
        }

        CountStatisticsTools2.clear();
        TimeStatisticsTools2.clear();
    }

    private void reportTime2Metric() {
        Map<String, StatTimeItem> countMap = TimeStatisticsTools2.getTimeStatMap();
        countMap.forEach((k, v) -> {
            WebApiStatisticsReportTools.reportTimeConsumed(v.getKey(), v.getCount(), v.avg());
            ConfigTools3.getAsList("statistics.level").forEach(s -> {
                WebApiStatisticsReportTools.reportTimeConsumedByStatus(s, v);
            });
        });
    }

    private void reportCount2Metric() {
        Map<String, Map<String, Long>> countMap = CountStatisticsTools2.getCountData();
        countMap.forEach((k, v) -> {
            v.forEach((k1, v1) -> {
                WebApiStatisticsReportTools.reportCount(String.format("%s-%s", k, k1), v1);
            });
        });
    }

    private void reportQpiQps2Metric() {
        Map<ServiceStatisticItem, QpsItem> visit = getAndResetVisit();
        if (!CollectionUtils.isNullOrEmpty(visit)){
            for (Map.Entry element : visit.entrySet()){
                reportQps(element);
            }
        }
    }
}
