package common.web.tools.filter.crypt;

import com.google.common.base.Strings;
import com.google.common.collect.Maps;
import common.base.tools.compress.GZipTools;
import common.base.tools.encrypt.AESUtils;
import common.base.tools.exception.ApiException;
import common.config.tools.config.ConfigTools3;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;
import java.util.Map;

/**
 * Created by Frank.Huang on 2017/5/16.
 */
public class CryptMsgTools {
    private static final Logger logger = LoggerFactory.getLogger(CryptMsgTools.class);

    public static String CRYPT_AES_CONFIG_KEY = "crypt.aes.key";
    public static String CRYPT_JAQ_CONFIG_KEY = "crypt.jaq.key";

    public static String CRYPT_JAQ_CONFIG_KEY_V2 = "crypt.jaq.v2.key";
    public static String CRYPT_JAQ_CONFIG_KEY_V2_AES_KEY = "crypt.jaq.v2.key.aes.key";
    public static String CRYPT_JAQ_CONFIG_KEY_V2_AES_IV = "crypt.jaq.v2.key.aes.iv";

    private static Map<String, String> cryptKeyMap = Maps.newHashMap();

    public static void setCryptKey(String key, String value) {
        cryptKeyMap.put(key, value);
    }

    public static String getCryptKey(String key) {
        String cryptKey = ConfigTools3.getString(key, cryptKeyMap.getOrDefault(key, ""));
        if (Strings.isNullOrEmpty(cryptKey)) {
            throw new RuntimeException(String.format("Crypt key[%s] not configuration!", key));
        }
        return cryptKey;

    }

    public static String getCryptJaqKey() {
        return getCryptKey(CRYPT_JAQ_CONFIG_KEY);
    }

    public static String getCryptJaqV2Key() {
        return getCryptKey(CRYPT_JAQ_CONFIG_KEY_V2);
    }

    public static String getCryptAesKey() {
        return getCryptKey(CRYPT_AES_CONFIG_KEY);
    }

    public static String decryptMsg(String protocol, String msg, String signature) throws Exception {
        return decryptMsg(protocol, msg, signature, false);
    }

    public static String decryptMsg(String protocol, String msg, String signature, boolean isSupport) throws Exception {
        if (Strings.isNullOrEmpty(msg)) {
            logger.error("Encrypt msg is empty");
            return "";
        }

        String ungzip = msg;
        if (isSupport && isSupportGzip(protocol)) {
            ungzip = GZipTools.decompress(msg);
        }

        try {
            if (isCryptAES(protocol)) {
                return AESUtils.decrypt(ungzip, getCryptAesKey());
            } else if (isCryptJAQ(protocol)) {

                String aesMsg = "";
                if (isCryptJAQV2(protocol)) {
                    //for jaq2,the msg is crypt bye aes msg=aes(jaq(payload))
                    aesMsg = ungzip;
                    msg = AESIVTools.decrypt(aesMsg, getCryptKey(CRYPT_JAQ_CONFIG_KEY_V2_AES_KEY), getCryptKey(CRYPT_JAQ_CONFIG_KEY_V2_AES_IV));
                    if (Strings.isNullOrEmpty(msg)) {
                        return "";
                    }
                }

                /**
                 * for tv check sign after decrypt,
                 * other check sign first
                 */
                String jaqKey = "";
                if (isCryptJAQV2(protocol)) {
                    jaqKey = getCryptJaqV2Key();
                } else {
                    jaqKey = getCryptJaqKey();
                }

                if (!isLiveMsg(protocol)) {
                    boolean checked = JaqSecurity.checkSign(msg, jaqKey, signature);
                    if (!checked) {
                        throw new ApiException(-999, 1, "Message sign check failed.");
                    }
                }

                byte[] decodeBytes = JaqSecurity.decrypt(msg.getBytes(), jaqKey);
                if (decodeBytes == null) {
                    return "";
                }

                String payload = new String(decodeBytes);
                if (isLiveMsg(protocol)) {
                    boolean checked = true;
                    if (isCryptJAQV2(protocol)) {
                        checked = JaqSecurity.checkSign(aesMsg, jaqKey, signature);
                    } else {
                        checked = JaqSecurity.checkSign(payload, jaqKey, signature);
                    }

                    if (!checked) {
                        throw new ApiException(-999, 1, "Message sign check failed.");
                    }
                }

                return payload;
            } else {
                throw new ApiException(-999, 2, "Message decrypt exception.");
            }
        } catch (Exception e) {
            logger.error("Decode arg exception:{}", e);
            if (e instanceof ApiException) {
                throw e;
            } else {
                throw new ApiException(-999, 3, "Message decrypt exception.");
            }
        }
    }

    public static String encryptMsg(String protocol, String msg) {
        if (Strings.isNullOrEmpty(msg)) {
            logger.error("Encrypt msg is empty");
            return "";
        }

        String gzipMsg = msg;
        if (isSupportGzip(protocol)) {
            gzipMsg = GZipTools.compress(msg);
        }

        try {
            if (isCryptAES(protocol)) {
                return AESUtils.encrypt(gzipMsg, getCryptAesKey());
            } else if (isCryptJAQ(protocol)) {
                if (isCryptJAQV2(protocol)) {
                    if (Strings.isNullOrEmpty(gzipMsg)) {
                        return "";
                    }

                    String jaqMsg = JaqSecurity.encrypt(gzipMsg, getCryptJaqV2Key());
                    if (Strings.isNullOrEmpty(jaqMsg)) {
                        return "";
                    }

                    return AESIVTools.encrypt(jaqMsg, getCryptKey(CRYPT_JAQ_CONFIG_KEY_V2_AES_KEY), getCryptKey(CRYPT_JAQ_CONFIG_KEY_V2_AES_IV));
                } else {
                    return JaqSecurity.encrypt(gzipMsg, getCryptJaqKey());
                }
            } else {
                return "";
            }
        } catch (Exception e) {
            logger.error("Encrypt Message exception:{}", e);
            return "";
        }
    }

    public static String encryptMsg(String protocol, byte[] msg) {
        return encryptMsg(protocol, new String(msg));
    }

    public static String sign(String protocol, String msg) {
        try {
            if (isCryptJAQ(protocol)) {
                if (isCryptJAQV2(protocol)) {
                    return JaqSecurity.sign(msg, getCryptJaqV2Key());
                } else {
                    return JaqSecurity.sign(msg, getCryptJaqKey());
                }
            } else return "";
        } catch (Exception e) {
            logger.error("Sign message exception:{}", e);
            return "";
        }
    }

    public static boolean isCryptJAQ(String v) {
        if (CryptProtocolVersion.PROT_4.contentEquals(v)) {
            return true;
        }

        if (CryptProtocolVersion.PROT_ACM11.contentEquals(v)) {
            return true;
        }

        if (CryptProtocolVersion.PROT_ACM20.contentEquals(v)) {
            return true;
        }

        if (CryptProtocolVersion.PROT_ACM21.contentEquals(v)) {
            return true;
        }

        return false;
    }

    public static boolean isCryptJAQV2(String v) {
        if (CryptProtocolVersion.PROT_ACM20.contentEquals(v)) {
            return true;
        }

        if (CryptProtocolVersion.PROT_ACM21.contentEquals(v)) {
            return true;
        }

        return false;
    }


    public static boolean isCryptAES(String v) {
        if (CryptProtocolVersion.PROT_3.contentEquals(v)) {
            return true;
        }

        if (CryptProtocolVersion.PROT_ACM1.contentEquals(v)) {
            return true;
        }

        return false;
    }


    public static boolean isLiveMsg(String v) {
        if (CryptProtocolVersion.PROT_3.contentEquals(v)
            || CryptProtocolVersion.PROT_4.contentEquals(v)) {
            return false;
        }

        return true;
    }

    public static boolean isSupportCrypt(String v) {
        return CryptProtocolVersion.PROT_3.contentEquals(v)
            || CryptProtocolVersion.PROT_4.contentEquals(v)
            || CryptProtocolVersion.PROT_ACM1.contentEquals(v)
            || CryptProtocolVersion.PROT_ACM11.contentEquals(v)
            || CryptProtocolVersion.PROT_ACM20.contentEquals(v)
            || CryptProtocolVersion.PROT_ACM21.contentEquals(v);
    }


    public static boolean isSupportGzip(String v) {
        return CryptProtocolVersion.PROT_ACM21.contentEquals(v);
    }

    public static boolean isSupportProtocol(String v) {
        List<String> protocols = ConfigTools3.getAsList("crypt.support.protocol");
        if (protocols.isEmpty()) {
            return true;
        }

        if (protocols.contains(v)) {
            return true;
        }

        return false;
    }


}