package common.web.tools.http.restClient;

import org.apache.http.client.HttpClient;
import org.apache.http.client.HttpRequestRetryHandler;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;
import org.apache.http.impl.nio.client.HttpAsyncClientBuilder;
import org.apache.http.protocol.HttpContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.client.ClientHttpRequestFactory;
import org.springframework.http.client.HttpComponentsAsyncClientHttpRequestFactory;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;

import java.io.IOException;

public class RestCallClient {
    private static Logger logger = LoggerFactory.getLogger(RestCallClient.class);

    private static int maxHttpConnectionTotal = 1000;
    private static int maxHttpConnectionPerRoute = 1000;
    private static int timeout = 60_1000;
    private static int failbackRetry = 1;

    public static void init(int maxHttpConnectionTotal, int maxHttpConnectionPerRoute, int timeout) {
        RestCallClient.maxHttpConnectionTotal = maxHttpConnectionTotal;
        RestCallClient.maxHttpConnectionPerRoute = maxHttpConnectionPerRoute;
        RestCallClient.timeout = timeout;
    }

    public static ClientHttpRequestFactory syncHttpFactory() {
        return new HttpComponentsClientHttpRequestFactory(getHttpSyncClient());
    }

    public static HttpComponentsAsyncClientHttpRequestFactory asyncHttpFactory() {
        return new HttpComponentsAsyncClientHttpRequestFactory(getHttpAsyncClient());
    }

    private static RequestConfig getRequestConfig() {
        if (timeout == 0) {
            timeout = 1000;
        }

        return RequestConfig.custom().setSocketTimeout(timeout).setConnectTimeout(timeout).setConnectionRequestTimeout(timeout).build();
    }

    private static PoolingHttpClientConnectionManager getConnectionManager() {
        Registry<ConnectionSocketFactory> registry = RegistryBuilder.<ConnectionSocketFactory>create().register("http", PlainConnectionSocketFactory.getSocketFactory()).build();
        PoolingHttpClientConnectionManager connectionManager = new PoolingHttpClientConnectionManager(registry);
        connectionManager.setMaxTotal(maxHttpConnectionTotal);
        connectionManager.setDefaultMaxPerRoute(maxHttpConnectionPerRoute);

        return connectionManager;
    }


    private static HttpClient getHttpSyncClient() {
        return HttpClientBuilder.create()
            .setDefaultRequestConfig(getRequestConfig())
            .setConnectionManager(getConnectionManager())
            .setRetryHandler(new HttpRequestRetryHandler() {
                @Override
                public boolean retryRequest(IOException exception, int executionCount, HttpContext context) {
                    logger.info("HttpClient failback Retry:{}:{}", executionCount, exception);
                    if (executionCount < failbackRetry) {
                        return true;
                    }
                    return false;
                }
            })
            .build();
    }


    private static CloseableHttpAsyncClient getHttpAsyncClient() {
        return HttpAsyncClientBuilder.create()
            .setDefaultRequestConfig(getRequestConfig())
            .setMaxConnTotal(maxHttpConnectionTotal)
            .setMaxConnPerRoute(maxHttpConnectionPerRoute)
            .build();
    }
}
