package common.web.tools.filter.common;

import com.google.common.base.Strings;
import com.google.gson.Gson;
import common.base.tools.type.CollectionUtils;
import org.springframework.http.HttpMethod;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;
import java.io.BufferedReader;
import java.io.IOException;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;

public class WebRequestWrapper extends HttpServletRequestWrapper {
    private Map<String, String[]> parameterMap = new HashMap<String, String[]>();
    private Gson gson = new Gson();

    public WebRequestWrapper(HttpServletRequest request) {
        super(request);
        if (HttpMethod.POST.name().equalsIgnoreCase(request.getMethod())) {
            getPostBody(request);
        }
        
        parameterMap.putAll(request.getParameterMap());
    }

    // 重写getParameter，代表参数从当前类中的map获取
    @Override
    public String getParameter(String name) {
        String[] values = parameterMap.get(name);
        if (values == null || values.length == 0) {
            return null;
        }
        return values[0];
    }

    // getParameterValues，代表参数从当前类中的map获取
    @Override
    public String[] getParameterValues(String name) {
        return parameterMap.get(name);
    }

    @Override
    public Map<String, String[]> getParameterMap() {
        return parameterMap;
    }

    @Override
    public Enumeration getParameterNames() {
        return Collections.enumeration(this.parameterMap.keySet());
    }

    public void addAllParameters(Map<String, Object> otherParams) {// 增加多个参数
        if (!CollectionUtils.isNullOrEmpty(otherParams)) {
            for (Map.Entry<String, Object> entry : otherParams.entrySet()) {
                addParameter(entry.getKey(), entry.getValue());
            }
        }
    }

    public void addParameter(String name, Object value) {// 增加参数
        if (value != null) {
            if (value instanceof String[]) {
                parameterMap.put(name, (String[]) value);
            } else if (value instanceof String) {
                parameterMap.put(name, new String[]{(String) value});
            } else {
                parameterMap.put(name, new String[]{String.valueOf(value)});
            }
        }
    }

    private void getPostBody(HttpServletRequest request) {
        String postBody = getPostBodyString(request);
        if (!Strings.isNullOrEmpty(postBody)) {
            Map<String, Object> postMap = gson.fromJson(postBody, Map.class);
            if (!CollectionUtils.isNullOrEmpty(postMap)) {
                postMap.forEach((k, v) -> {
                    addParameter(k, v);
                });
            }
        }
    }

    public static String getPostBodyString(HttpServletRequest request) {

        BufferedReader br = null;
        StringBuilder sb = new StringBuilder("");
        try {
            br = request.getReader();
            String str;
            while ((str = br.readLine()) != null) {
                sb.append(str);
            }
            br.close();
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            if (null != br) {
                try {
                    br.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        return sb.toString();
    }

}
