package common.web.tools.statistics.scheduler;

import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import common.base.tools.stat.StatTools;
import common.base.tools.statistics.CountStatisticsTools2;
import common.base.tools.statistics.MetricTools2;
import common.base.tools.statistics.StatTimeItem;
import common.base.tools.statistics.TimeStatisticsTools2;
import common.base.tools.sys.SystemTools;
import common.config.tools.config.ConfigTools3;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Service;

import java.util.Map;

@Service
public class WebApiStatisticsScheduler {

    private static final Logger logger = LoggerFactory.getLogger(WebApiStatisticsScheduler.class);
    static long lastReportTS = 0;

    @Scheduled(fixedRate = 3000L)
    public void statisticsScheduler() {
        long cycle = ConfigTools3.getLong("api.statistics.cycle", 60 * 1000L);
        if (System.currentTimeMillis() - lastReportTS > cycle) {
            reportAndPrint();
            lastReportTS = System.currentTimeMillis();
        }
    }

    @Scheduled(fixedDelay = 60000)
    public void localStatisticsScheduler() {
        String msg = StatTools.dumpMutilline(false);
        if (!Strings.isNullOrEmpty(msg)) {
            logger.info("LOCAL-STAT:{}", msg);
        }
    }

    public void reportAndPrint() {
        if (ConfigTools3.getBoolean("statistics.out.log", true)) {
            boolean printPretty = ConfigTools3.getBoolean("statistics.out.pretty", true);
            String countStats = CountStatisticsTools2.dump(printPretty);
            String timeStats = TimeStatisticsTools2.dump(printPretty);

            if (!Strings.isNullOrEmpty(countStats) || !Strings.isNullOrEmpty(timeStats)) {
                logger.info("CountStats:{} TimeStats:{}", countStats, timeStats);
            }
        }

        if (ConfigTools3.getBoolean("statistics.report", false)) {
            reportCount2Metric();
            reportTime2Metric();
        }

        CountStatisticsTools2.clear();
        TimeStatisticsTools2.clear();
    }

    private void reportTime2Metric() {
        Map<String, StatTimeItem> countMap = TimeStatisticsTools2.getTimeStatMap();
        countMap.forEach((k, v) -> {
            reportTimeConsumed(v.getKey(), v.getCount(), v.avg());
            ConfigTools3.getAsList("statistics.level").forEach(s -> {
                reportTimeConsumedByStatus(s, v);
            });
        });
    }


    private void reportCount2Metric() {
        Map<String, Map<String, Long>> countMap = CountStatisticsTools2.getCountData();
        countMap.forEach((k, v) -> {
            v.forEach((k1, v1) -> {
                reportCount(String.format("%s-%s", k, k1), v1);
            });
        });
    }

    private void reportCount(String key, long count) {
        String metricName = String.format("%s_count", ConfigTools3.getString("statistics.metric.prefix", "common_statistics"));
        ImmutableMap tags = ImmutableMap.of("host", SystemTools.getHostname(),
            "item", key);
        ImmutableMap values = ImmutableMap.of("count", count);
        MetricTools2.postEvent(metricName, tags, values);
    }

    private void reportTimeConsumed(String key, long count, long avg) {
        String metricName = String.format("%s_time_consumed", ConfigTools3.getString("statistics.metric.prefix", "common_statistics"));
        ImmutableMap tags = ImmutableMap.of("host", SystemTools.getHostname(),
            "item", key);
        ImmutableMap values = ImmutableMap.of("count", count, "avg", avg);
        MetricTools2.postEvent(metricName, tags, values);
    }

    private void reportTimeConsumedByStatus(String key, StatTimeItem timeItem) {
        int level = ConfigTools3.getInt(String.format("statistics.level.%s", key), 0);
        if (level <= 0) {
            return;
        }

        long count = timeItem.getCountGELevel(level);
        if (count == 0) {
            return;
        }

        String reportKey = String.format("%s.%s", timeItem.getKey(), key);

        String metricName = String.format("%s_time_status", ConfigTools3.getString("statistics.metric.prefix", "common_statistics"));
        ImmutableMap tags = ImmutableMap.of("host", SystemTools.getHostname(),
            "item", reportKey);
        ImmutableMap values = ImmutableMap.of("count", count);
        MetricTools2.postEvent(metricName, tags, values);
    }
}
