package common.diagnosis.springboot.starter.buildinfo;

import com.google.common.collect.Maps;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.info.BuildProperties;
import org.springframework.boot.info.GitProperties;

import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.stream.Collectors;


public class BuildInfoProperties {
    private final Logger logger = LoggerFactory.getLogger(BuildInfoProperties.class);
    private Map<String, String> properties = Maps.newLinkedHashMap();

    public BuildInfoProperties(GitProperties gitProperties, BuildProperties buildProperties) {
        //build info
        if (Objects.nonNull(buildProperties)) {
            properties.put("package.build.name", buildProperties.getArtifact());
            properties.put("package.build.group", buildProperties.getGroup());
            properties.put("package.build.artifact", buildProperties.getArtifact());
            properties.put("package.build.version", buildProperties.getVersion());
            properties.put("package.build.time", formatInstant(buildProperties.getTime()));
        }

        //code info(git)
        if (Objects.nonNull(gitProperties)) {
            properties.put("code.commit.branch", gitProperties.getBranch());
            properties.put("code.commit.id", gitProperties.getShortCommitId());
            properties.put("code.commit.user.name", gitProperties.get("commit.user.name"));
            properties.put("code.commit.time", formatInstant(gitProperties.getCommitTime()));
            properties.put("code.build.time", formatInstant(gitProperties.getInstant("build.time")));
            properties.put("code.build.host", gitProperties.get("build.host"));
        }

        //runtime info
        System.getProperties().entrySet().forEach(e -> {
            String key = (String) e.getKey();
            if (key.startsWith("java.vm.")
                    || key.startsWith("java.runtime.")
                    || key.startsWith("java.specification.")
                    || key.startsWith("os.")
                    || key.startsWith("user.country")
                    || key.startsWith("user.timezone")) {
                properties.put(String.format("runtime.%s", key), (String) e.getValue());
            }
        });

        //print info
        printfBuildInfoProperties();
    }

    public Map<String, String> buildInfoProperties() {
        return properties;
    }

    public Map<String, String> buildInfoProperties(String prefix) {
        return properties.entrySet().stream().filter(e -> e.getKey().startsWith(prefix))
                .collect(Collectors.toMap(e -> e.getKey(), e -> e.getValue()));
    }

    public void printfBuildInfoProperties() {
        properties.entrySet().forEach(e -> {
            logger.info("{}={}", e.getKey(), e.getValue());
        });
    }

    private String formatInstant(Instant instant) {
        return DateTimeFormatter.ISO_OFFSET_DATE_TIME.withZone(ZoneId.of("Asia/Shanghai")).format(instant);
    }

}
