package common.diagnosis.springboot.starter.tools;

import com.google.common.base.Charsets;
import com.google.common.base.Joiner;
import com.google.common.base.Strings;
import com.google.common.collect.Sets;
import com.google.common.hash.Hashing;
import com.google.common.io.BaseEncoding;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.lang.management.ManagementFactory;
import java.lang.management.RuntimeMXBean;
import java.net.*;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Enumeration;
import java.util.Set;


/**
 * System level helper methods
 *
 * @author Stuart
 */
public class SystemTools {
    private static final Logger logger = LoggerFactory.getLogger(SystemTools.class);
    private static String ipv4 = null;
    private static String ipv6 = null;
    private static String hostname = "";
    private static String tmpDir = "";

    private SystemTools() {
        throw new UnsupportedOperationException("Utility class");
    }

    /**
     * Search through the the network adapters and get the IPv4 address of the
     * server.
     *
     * @return
     */
    public static String getIpAddress() {
        return getIpAddress(Boolean.TRUE);
    }

    /**
     * Search through the network adapters and get the IP address of the server.
     *
     * @param getIpv4 return the IPv4 address, otherwise IPv6
     * @return
     */
    public static String getIpAddress(boolean getIpv4) {
        if (getIpv4 && ipv4 != null) {
            return ipv4;
        }

        if (!getIpv4 && ipv6 != null) {
            return ipv6;
        }

        try {
            Enumeration<NetworkInterface> interfaces = NetworkInterface.getNetworkInterfaces();
            boolean found = Boolean.FALSE;
            while (!found && interfaces.hasMoreElements()) {
                NetworkInterface currentInterface = interfaces.nextElement();
                if (!currentInterface.isUp() || currentInterface.isLoopback() || currentInterface.isVirtual()) {
                    continue;
                }

                logger.trace("Current Inteface: {}", currentInterface.toString());
                Enumeration<InetAddress> addresses = currentInterface.getInetAddresses();
                while (!found && addresses.hasMoreElements()) {
                    InetAddress currentAddress = addresses.nextElement();
                    if (currentAddress.isLoopbackAddress()) {
                        continue;
                    }

                    if (currentAddress instanceof Inet4Address) {
                        ipv4 = currentAddress.getHostAddress();
                        logger.trace("IPv4 Address: {}", currentAddress.getHostAddress());
                    } else if (currentAddress instanceof Inet6Address) {
                        ipv6 = currentAddress.getHostAddress();
                        logger.trace("IPv6 Address: {}", currentAddress.getHostAddress());
                    }

                    if (ipv4 != null && ipv6 != null) {
                        found = Boolean.TRUE;
                    }
                }
            }
        } catch (SocketException ex) {
            logger.error("Failed to get IP Address, error: {}", ex.getMessage());
        }

        if (getIpv4) {
            return ipv4;
        }
        return ipv6;
    }

    public static Set<String> getIpAddressList() {
        Set<String> ips = Sets.newHashSet();
        try {
            Enumeration<NetworkInterface> interfaces = NetworkInterface.getNetworkInterfaces();
            boolean found = Boolean.FALSE;
            while (!found && interfaces.hasMoreElements()) {
                NetworkInterface currentInterface = interfaces.nextElement();
                if (!currentInterface.isUp() || currentInterface.isLoopback() || currentInterface.isVirtual()) {
                    continue;
                }

                Enumeration<InetAddress> addresses = currentInterface.getInetAddresses();
                while (!found && addresses.hasMoreElements()) {
                    InetAddress currentAddress = addresses.nextElement();
                    if (currentAddress.isLoopbackAddress()) {
                        continue;
                    }

                    if (currentAddress instanceof Inet4Address) {
                        String ip = currentAddress.getHostAddress();
                        if (!Strings.isNullOrEmpty(ip)) {
                            ips.add(ip);
                        }
                    }
                }
            }
        } catch (SocketException ex) {
            logger.error("Failed to get IP Address, error: {}", ex.getMessage());
        }

        return ips;
    }

    public static String getIpAddress(boolean getIpv4, String networkInterfaceName) {

        try {
            Enumeration<NetworkInterface> interfaces = NetworkInterface.getNetworkInterfaces();
            boolean found = Boolean.FALSE;
            while (!found && interfaces.hasMoreElements()) {
                NetworkInterface currentInterface = interfaces.nextElement();
                if (!currentInterface.isUp() || currentInterface.isLoopback() || currentInterface.isVirtual()) {
                    continue;
                }

                if (!networkInterfaceName.equals(currentInterface.getDisplayName())) {
                    continue;
                }
                logger.trace("Current Inteface: {}", currentInterface.toString());
                Enumeration<InetAddress> addresses = currentInterface.getInetAddresses();
                while (!found && addresses.hasMoreElements()) {
                    InetAddress currentAddress = addresses.nextElement();
                    if (currentAddress.isLoopbackAddress()) {
                        continue;
                    }

                    if (currentAddress instanceof Inet4Address && getIpv4) {
                        logger.trace("IPv4 Address: {}:{}", networkInterfaceName, currentAddress.getHostAddress());
                        return currentAddress.getHostAddress();

                    } else if (currentAddress instanceof Inet6Address & !getIpv4) {
                        logger.trace("IPv6 Address: {}:{}", networkInterfaceName, currentAddress.getHostAddress());
                        return currentAddress.getHostAddress();
                    }
                }
            }
        } catch (SocketException ex) {
            logger.error("Failed to get IP Address, error: {}", ex.getMessage());
        }

        return "";
    }


    /**
     * 获取当前系统换行符
     */
    public static String LINE_SEPARATOR = System.getProperty("line.separator");

    /**
     * 获取当前线程id
     *
     * @return
     */
    public static long currentThreadId() {
        return Thread.currentThread().getId();
    }

    public static String getHostnameMD5() {
        String host = getHostname();
        if (Strings.isNullOrEmpty(host)) {
            return "";
        }

        return Hashing.md5().hashString(host, Charset.forName("utf-8")).toString();
    }

    public static String getHostNameAES() {
        String host = getHostname();
        if (Strings.isNullOrEmpty(host)) {
            return "";
        }

        byte[] bytes = AESIVTools.encrypt(host.getBytes(Charsets.UTF_8),
                "io.ziyi.vod.2021".getBytes(Charsets.UTF_8),
                "+860755-905-9527".getBytes(Charsets.UTF_8));
        return BaseEncoding.base16().encode(bytes);
    }

    public static String getHostname() {
        if (!Strings.isNullOrEmpty(hostname)) {
            return hostname;
        }

        try {
            hostname = InetAddress.getLocalHost().getHostName();
            if (!Strings.isNullOrEmpty(hostname)) {
                return hostname;
            }
        } catch (UnknownHostException e) {
            // failed;  try alternate means.
        }

        // try environment properties.
        hostname = System.getenv("COMPUTERNAME");
        if (!Strings.isNullOrEmpty(hostname)) {
            return hostname;
        }

        hostname = System.getenv("HOSTNAME");
        if (!Strings.isNullOrEmpty(hostname)) {
            return hostname;
        }

        // undetermined.
        return null;
    }

    public static String getTmpDir() {
        return getTmpDir("");
    }

    public static String getTmpDir(String path) {
        if (!Strings.isNullOrEmpty(tmpDir)) {
            return tmpDir;
        }

        tmpDir = System.getProperty("java.io.tmpdir");
        if (Strings.isNullOrEmpty(tmpDir)) {
            tmpDir = "./tmp";
        }
        if (tmpDir.endsWith(File.separator)) {
            tmpDir = tmpDir + "mfc-vms";
        } else {
            tmpDir = tmpDir + File.separator + "mfc-vms";
        }

        if (!Strings.isNullOrEmpty(path)) {
            tmpDir = tmpDir + File.separator + path;
        }

        int ret = mkdirIfAbsent(tmpDir);
        if (ret == 0) {
            logger.info("mkdir:{} success", tmpDir);
        } else {
            logger.info("mkdir:{} failed", tmpDir);
        }

        return tmpDir;
    }

    public static int mkdirIfAbsent(String dir) {
        if (Strings.isNullOrEmpty(dir)) {
            return 1;
        }

        File folder = new File(dir);
        if (folder.exists() && folder.isDirectory()) {
            return 0;
        }

        if (folder.mkdirs()) {
            return 0;
        } else {
            return 1;
        }
    }

    public static final int getCurrentPid() {
        RuntimeMXBean runtimeMXBean = ManagementFactory.getRuntimeMXBean();
        String processName = runtimeMXBean.getName();
        if (processName.indexOf('@') != -1) {
            String pid = processName.substring(0, processName.indexOf('@'));
            if (!Strings.isNullOrEmpty(pid)) {
                return Integer.valueOf(pid);
            }
        }

        return -1;
    }

    public static void printEnvs() {
        printEnvs(null);
    }

    public static void printEnvs(String appdir) {
        if (Strings.isNullOrEmpty(appdir)) {
            appdir = "mfc-vms-tmp";
        }
        String tmpdir = SystemTools.getTmpDir(appdir);
        logger.info("******************************************************************************");
        logger.info("*IP             :{}", Joiner.on(",").skipNulls().join(SystemTools.getIpAddressList()));
        logger.info("*HOST           :{}({})", SystemTools.getHostname(), SystemTools.getHostnameMD5());
        logger.info("*TMP DIR        :{}", tmpdir);
        logger.info("*PID            :{}", SystemTools.getCurrentPid());
        logger.info("******************************************************************************");


    }

    public static String getenv(String key) {
        return System.getenv(key);
    }

}
