package com.valor.common.flowcontrol.client.service.impl.flowcontrol;

import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.cache.Cache;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.valor.common.flowcontrol.client.constant.EFlowControlResultCode;
import com.valor.common.flowcontrol.client.model.*;
import com.valor.common.flowcontrol.client.service.impl.api.FlowControlApi;
import com.valor.common.flowcontrol.client.service.impl.api.IFlowControlApi;
import com.valor.common.flowcontrol.client.service.impl.cache.CacheFactory;
import com.valor.common.flowcontrol.client.service.ResponseBuilder;
import common.config.tools.config.loader.json.JsonSerialize;
import common.web.tools.http.model.response.WebApiBaseResponse;
import common.web.tools.http.model.response.WebApiResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.util.Map;
import java.util.Set;

/**
 * @author Hunter Chen created on: 2018-09-10
 */
public class FlowControlClient {
    private static Logger logger = LoggerFactory.getLogger(FlowControlClient.class);

    private static final int INIT_STATUS_SUCCESS = 0;
    private static final int INIT_STATUS_FAILED = 1;
    private static final int INIT_STATUS_EXCEPTION = 2;
    private static int initStatus = INIT_STATUS_FAILED;
    private static int initCount = 0;

    private static IFlowControlApi flowControlApi = null;
    private static FlowControlConfig config = null;
    private static Map<String, FlowControlConfigItem> FlowControlConfigMap = Maps.newHashMap();
    private static String configFileName;


    /**
     * Initialization method. Only needs to be initialized once each deployment. Recommend
     * initialization during server start up.
     */
    public static WebApiBaseResponse init(String configFile) {
        configFileName = configFile;

        loadConfig(configFileName);
        if (config == null) {
            logger.error("Flow control not configured properly! Please fill in all the required configuration");

            return ResponseBuilder.buildResponse(EFlowControlResultCode.INIT_ERROR);
        }

        config.getServer().getFlowControlList().forEach(e -> {
            FlowControlConfigMap.put(e.getKey(), e);
            logger.info(e.toString());
        });

        //init cache
        CacheFactory.initCache(config);

        WebApiBaseResponse response = serverInit();
        return response;
    }

    /**
     * counter
     *
     * @param key
     * @param value
     * @param count
     * @return
     */
    public static WebApiBaseResponse counter(String key, String value, int count) {
        WebApiBaseResponse response = isAlreadyOutOfLimit(key, value);
        if (!response.isSuccessful()) {
            return response;
        }

        //add to local cache
        int size = addCounter(key, value, count, getBatchSize(key));
        if (size <= 0) {
            return ResponseBuilder.buildResponse(EFlowControlResultCode.SUCCESS);
        }

        //build request
        FlowControlCounterReq counterReq = new FlowControlCounterReq();
        counterReq.setAppId(getConfig().getServer().getAppId());
        counterReq.setAppKey(getConfig().getServer().getAppKey());
        counterReq.setKey(key);
        counterReq.setValue(value);
        counterReq.setCount(size);

        response = checkServerInit();
        if (!response.isSuccessful()) {
            return response;
        }

        //async
        FlowControlConfigItem configItem = FlowControlConfigMap.get(key);
        if (configItem.getAsync() == 1) {
            flowControlApi.counterAsync(counterReq, new FlowControlAsyncCallBack());
            return ResponseBuilder.buildResponse(EFlowControlResultCode.SUCCESS);
        } else {
            //sync
            WebApiResponse<FlowControlBaseRsp> fcsResponse = flowControlApi.counter(counterReq);
            String fcKey = FlowControlKeyBuilder.buildKey(getIntervalSeconds(key), key, value);
            return getWebApiBaseResponse(fcKey, fcsResponse);
        }
    }

    public static WebApiBaseResponse counter(String key, String value) {
        return counter(key, value, 1);
    }


    public static WebApiBaseResponse counterAsync(FlowControlCounterReq counterReq) {
        WebApiBaseResponse response = checkServerInit();
        if (!response.isSuccessful()) {
            return response;
        }

        //async
        flowControlApi.counterAsync(counterReq, new FlowControlAsyncCallBack());
        return ResponseBuilder.buildResponse(EFlowControlResultCode.SUCCESS);
    }

    public static WebApiBaseResponse uniqueCounterAsync(FlowControlUniqueCounterReq counterReq) {
        WebApiBaseResponse response = checkServerInit();
        if (!response.isSuccessful()) {
            return response;
        }

        //async
        flowControlApi.uniqueCounterAsync(counterReq, new FlowControlAsyncCallBack());
        return ResponseBuilder.buildResponse(EFlowControlResultCode.SUCCESS);
    }


    public static WebApiBaseResponse uniqueCounter(String l1Key, String l2Key, Set<String> valueSet) {
        WebApiBaseResponse response = isAlreadyOutOfLimit(l1Key, l2Key);
        if (!response.isSuccessful()) {
            return response;
        }

        //add to local cache
        Set<String> uniqSet = addUniqueCounter(l1Key, l2Key, valueSet, getBatchSize(l1Key));
        if (uniqSet.isEmpty()) {
            return ResponseBuilder.buildResponse(EFlowControlResultCode.SUCCESS);
        }

        response = checkServerInit();
        if (!response.isSuccessful()) {
            return response;
        }

        //build request
        FlowControlUniqueCounterReq uniqueCounter = new FlowControlUniqueCounterReq();
        uniqueCounter.setAppId(getConfig().getServer().getAppId());
        uniqueCounter.setAppId(getConfig().getServer().getAppKey());
        uniqueCounter.setKey(l1Key);
        uniqueCounter.setValue(l2Key);
        uniqueCounter.setCount(uniqSet);

        //async
        FlowControlConfigItem configItem = FlowControlConfigMap.get(l1Key);
        if (configItem.getAsync() == 1) {
            flowControlApi.uniqueCounterAsync(uniqueCounter, new FlowControlAsyncCallBack());
            return ResponseBuilder.buildResponse(EFlowControlResultCode.SUCCESS);
        } else {
            //sync
            WebApiResponse<FlowControlBaseRsp> fcsResponse = flowControlApi.uniqueCounter(uniqueCounter);
            String fcKey = FlowControlKeyBuilder.buildKey(getIntervalSeconds(l1Key), l1Key, l2Key);
            return getWebApiBaseResponse(fcKey, fcsResponse);
        }
    }

    public static WebApiBaseResponse uniqueCounter(String l1Key, String l2Key, String value) {
        return uniqueCounter(l1Key, l2Key, Sets.newHashSet(value));
    }


    private static WebApiBaseResponse getWebApiBaseResponse(String fcKey, WebApiResponse<FlowControlBaseRsp> fcsResponse) {
        if (fcsResponse.isSuccessful()) {
            return ResponseBuilder.buildResponse(EFlowControlResultCode.SUCCESS);
        } else {
            if (isOutOfLimit(fcsResponse)) {
                //add to limit
                CacheFactory.addOutOfLimit(fcKey);
                return ResponseBuilder.buildResponse(EFlowControlResultCode.OUT_OF_LIMIT);
            } else {
                return fcsResponse;
            }
        }
    }


    public static FlowControlConfig getConfig() {
        return config;
    }

    public static void setConfig(FlowControlConfig config) {
        FlowControlClient.config = config;
    }

    ///////////////////////////////////////////////////////
    private static FlowControlConfig loadConfig(String configFile) {
        TypeReference typeReference = new TypeReference<FlowControlConfig>() {
        };
        config = (FlowControlConfig) JsonSerialize.loadJson(new File(configFile), typeReference);
        logger.info("Load config:{}", config);
        return config;
    }

    public static boolean isOutOfLimit(WebApiBaseResponse response) {
        if (response.getErrCode() == EFlowControlResultCode.OUT_OF_LIMIT.getErrorCode()) {
            return true;
        } else {
            return false;
        }
    }

    public static int getIntervalSeconds(String key) {
        FlowControlConfigItem item = FlowControlConfigMap.get(key);
        if (item == null) {
            return -1;
        }

        return item.getIntervalSeconds();
    }

    public static int addCounter(String key, String value, int countSize, int batchSize) {
        Cache<String, Integer> cache = CacheFactory.getCache(key);
        if (cache == null) {
            return -1;
        }

        Integer count = cache.getIfPresent(value);
        if (count == null) {
            count = countSize;
        } else {
            count += countSize;
        }


        if (count < batchSize) {
            cache.put(value, count);
            return 0;
        } else {
            cache.invalidate(value);
            return count;
        }
    }

    public static Set<String> addUniqueCounter(String l1Key, String l2Key, Set<String> valueSet, int batchSize) {
        Cache<String, Set<String>> cache = CacheFactory.getCache(l1Key);
        if (cache == null) {
            return Sets.newHashSet();
        }

        Set<String> uniqSet = cache.getIfPresent(l2Key);
        if (uniqSet == null) {
            uniqSet = Sets.newHashSet();
        }

        uniqSet.addAll(valueSet);

        if (uniqSet.size() >= batchSize) {
            cache.invalidate(l2Key);
            return uniqSet;
        } else {
            cache.put(l1Key, uniqSet);
            return Sets.newHashSet();
        }
    }


    public static int getBatchSize(String key) {
        FlowControlConfigItem configItem = FlowControlConfigMap.get(key);
        if (configItem == null) {
            return -1;
        }

        int batchSize = configItem.getLimit();

        batchSize /= config.getClient().getBatchSizeRatio();
        if (batchSize <= 0) {
            batchSize = 1;
        }

        return batchSize;
    }

    public static String getFlowControlApi(String apiPath) {
        if (FlowControlConfigMap.keySet().contains(apiPath)) {
            return apiPath;
        }

        String allModuleApi = String.format("api-%s", config.getServer().getAppId());
        if (FlowControlConfigMap.keySet().contains(allModuleApi)) {
            return allModuleApi;
        }

        return "";
    }

    private static WebApiBaseResponse serverInit() {
        flowControlApi = new FlowControlApi(config.getClient());
        WebApiBaseResponse response = flowControlApi.init(config.getServer());
        if (!response.isSuccessful()) {
            if (response.getErrCode() == EFlowControlResultCode.HTTP_ERROR.getErrorCode()) {
                initStatus = INIT_STATUS_EXCEPTION;
                initCount += 1;
            } else {
                initStatus = INIT_STATUS_FAILED;
            }
        }

        initStatus = INIT_STATUS_SUCCESS;
        logger.info("Call to flowcontrol server init:{}", response);

        return response;
    }

    //check server init proper
    private static WebApiBaseResponse checkServerInit() {
        //if not init failed,return direct
        if (initStatus == INIT_STATUS_FAILED) {
            return ResponseBuilder.buildResponse(EFlowControlResultCode.INIT_ERROR);
        }

        //if connect exception,reconnect to server
        if (initStatus == INIT_STATUS_EXCEPTION && initCount < 100) {
            WebApiBaseResponse response = serverInit();
            if (!response.isSuccessful()) {
                return response;
            }
        }

        return ResponseBuilder.buildResponse(EFlowControlResultCode.SUCCESS);
    }

    //check if already out of limit
    public static WebApiBaseResponse isAlreadyOutOfLimit(String key, String value) {
        if (!FlowControlConfigMap.containsKey(key)) {
            return ResponseBuilder.buildResponse(EFlowControlResultCode.NOT_CONFIG);
        }

        //check if already out of limit
        String fcKey = FlowControlKeyBuilder.buildKey(getIntervalSeconds(key), key, value);
        if (CacheFactory.isOutOfLimit(fcKey)) {
            return ResponseBuilder.buildResponse(EFlowControlResultCode.OUT_OF_LIMIT);
        }

        return ResponseBuilder.buildResponse(EFlowControlResultCode.SUCCESS);
    }

}
