package common.metrics.client;


import com.google.common.base.Strings;
import common.metrics.client.http.ProtoHttpClientFactroy;
import common.metrics.client.utils.ExecutorThreadPool;
import common.metrics.model.ERetCode;
import common.metrics.model.protoAutoGenerate.MetricsEvent;
import common.metrics.model.protoAutoGenerate.MetricsRequest;
import common.metrics.model.protoAutoGenerate.MetricsResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;

public class MetricsClient implements IMetricsClient {
    private Logger logger = LoggerFactory.getLogger(MetricsClient.class);

    private MetricsConfig config;
    private BlockingQueue<MetricsEvent> eventQueue = null;
    private static ExecutorThreadPool threadPool;

    private MetricsClient(MetricsConfig config) {
        this.config = config;
        if (getStatus()) {
            if (Objects.isNull(eventQueue)) {
                eventQueue = new LinkedBlockingQueue<>(config.getQueueSize());
            }

            threadPool = new ExecutorThreadPool(Math.max(config.getThreadPoolSize(), 10));
            threadPool.start();
            threadPool.schedule(new EventConsumer(), 100L, 10L, TimeUnit.MILLISECONDS);

        } else {
            logger.warn("MetricsClient config is NOT Complete.");
        }
    }

    public boolean getStatus() {
        return this.config != null && this.config.isConfig();
    }

    @Override
    public ERetCode sendEvents(List<MetricsEvent> events) {
        if (!getStatus()) {
            return ERetCode.OK;
        }

        try {
            if (eventQueue.size() >= config.getQueueSize()) {
                return ERetCode.RET_CLIENT_OVERFLOW;
            }

            events.forEach(e -> {
                eventQueue.add(e);
                if (eventQueue.size() > config.getPostBatchSize()) {
                    threadPool.background(new EventConsumer());
                }
            });

        } catch (Exception e) {
            return ERetCode.RET_CLIENT_OVERFLOW;
        }

        return ERetCode.OK;
    }

    private class EventConsumer implements Runnable {
        @Override
        public void run() {
            List<MetricsEvent> entryList = new LinkedList<>();
            MetricsEvent entry = null;
            try {
                while ((entry = eventQueue.poll()) != null) {
                    entryList.add(entry);
                    if (entryList.size() >= config.getPostBatchSize()) {
                        break;
                    }
                }
            } catch (Exception ex) {
                logger.error("MetricsClient:Failed to get events from queue.", ex);
            }

            if (entryList.size() > 0) {
                try {
                    send2Server(entryList);
                    logger.debug("MetricsClient: request sent: count={0}", entryList.size());

                } catch (Exception e) {
                    logger.error("MetricsClient:Send to server failed!.", e);
                }
            }
        }
    }

    private void send2Server(List<MetricsEvent> events) {
        if (!getStatus()) {
            return;
        }
        MetricsRequest request = MetricsRequest.newBuilder()
                .setAppId(config.getAppId())
                .setAppKey(config.getAppKey())
                .addAllEvents(events)
                .build();

        MetricsResponse response = ProtoHttpClientFactroy.protoClientOverRetrofit(config.getHttpHosts(), config.getHttpConnectTimeout(), config.getHttpReadTimeout()).postEvent(request);
        logger.info("Metrics Send Result:[{}:{}] Size:[{}] Host:[{}]", response.getRet(), response.getMsg(), events.size(), config.getHttpHosts());
    }


    public final static class Builder {
        private MetricsClient instance = null;

        public MetricsClient build(MetricsConfig config) {
            if (Objects.isNull(instance)) {
                instance = new MetricsClient(config);
            }
            return instance;
        }
    }

}
