package common.metrics.micrometer.registry;

import com.google.common.collect.Lists;
import common.metrics.client.MetricsClient;
import common.metrics.client.MetricsConfig;
import common.metrics.model.ERetCode;
import common.metrics.model.protoAutoGenerate.FieldValue;
import common.metrics.model.protoAutoGenerate.MetricsEvent;
import io.micrometer.core.instrument.Meter;
import io.micrometer.core.instrument.Timer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;


import static common.metrics.micrometer.registry.MetricsMeterFormat.measureValueToLong;

public class MetricMeterSender {
    private Logger logger = LoggerFactory.getLogger(MetricMeterSender.class);
    private MetricsClient client;

    public MetricMeterSender(IMetricsMeterRegistryConfig config) {
        MetricsConfig metricsConfig = new MetricsConfig.Builder().build();
        metricsConfig.setAppId(config.getAppId())
                .setAppKey(config.getAppKey())
                .setQueueSize(config.clientQueueSize())
                .setPostBatchSize(config.clientBatchSize())
                .setThreadPoolSize(config.clientThreadSize())
                .setHttpHosts(config.getHost())
                .setHttpConnectTimeout(config.httpConnectTimeout())
                .setHttpReadTimeout(config.httpReadTimeout());

        client = new MetricsClient.Builder().build(metricsConfig);
    }

    public void sendReport(List<Meter> meters) {
        List<MetricsEvent> events = buildMetricsEvent(meters);
        if (events.size() > 0) {
            ERetCode ret = client.sendEvents(events);
            if (!ret.isSuccess()) {
                logger.warn("Send Event failed! ret=[{}]", ret);
            }
        }
    }

    private List<MetricsEvent> buildMetricsEvent(List<Meter> meters) {
        List<MetricsEvent> events = Lists.newLinkedList();
        for (Meter meter : meters) {
            MetricsEvent.Builder builder = MetricsEvent.newBuilder();
            builder.setEventId(meter.getId().getName());
            meter.getId().getTags().forEach(e -> {
                builder.putTags(e.getKey(), e.getValue());
            });

            meter.measure().forEach(e -> {
                builder.addFields(FieldValue.newBuilder()
                        .setKey(e.getStatistic().name())
                        .setNumValue(measureValueToLong(e.getValue()))
                        .build());
            });

            if (meter instanceof Timer) {
                long mean = measureValueToLong(((Timer) meter).mean(TimeUnit.MILLISECONDS));
                builder.addFields(FieldValue.newBuilder()
                        .setKey("MEAN")
                        .setNumValue(mean)
                        .build());
            }

            events.add(builder.build());
        }

        return events;
    }

}
