package common.metrics.micrometer.registry;

import com.google.common.base.Strings;
import common.log.Logging;
import de.vandermeer.asciitable.AsciiTable;
import de.vandermeer.asciitable.CWC_LongestLine;
import de.vandermeer.asciithemes.TA_GridThemes;
import de.vandermeer.asciithemes.a7.A7_Frames;
import de.vandermeer.asciithemes.a7.A7_Grids;
import de.vandermeer.asciithemes.a8.A8_Grids;
import de.vandermeer.skb.interfaces.document.TableRowStyle;
import io.micrometer.core.instrument.*;

import java.awt.*;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * Meter format function
 */
public class MetricsMeterFormat {

    public static String format(Meter meter) {
        if (Objects.isNull(meter)) {
            return "";
        }

        StringBuilder sb = new StringBuilder();

        sb.append("type=").append(meter.getId().getType().name()).append(" ");
        sb.append("name=").append(meter.getId().getName()).append(" ");

        String tagStr = formatTags(meter.getId().getTags());
        if (!Strings.isNullOrEmpty(tagStr)) {
            sb.append(tagStr).append(" ");
        }
        sb.append(formatMeasure(meter.measure()));
        if (meter instanceof Timer) {
            double mean = ((Timer) meter).mean(TimeUnit.MILLISECONDS);
            sb.append("AVG=").append(Double.valueOf(mean).longValue()).append(" ");
        }
        return sb.toString();
    }

    public static String formatTags(List<Tag> tags) {
        StringBuilder sb = new StringBuilder();
        tags.forEach(e -> {
            sb.append(e.getKey()).append("=").append(e.getValue()).append(" ");
        });
        return sb.toString();

    }

    public static String formatMeasure(Iterable<Measurement> it) {
        StringBuilder sb = new StringBuilder();
        it.forEach(m -> {
            if (m.getStatistic() != Statistic.TOTAL_TIME) {
                sb.append(m.getStatistic().name()).append("=").append(Double.valueOf(m.getValue()).longValue()).append(" ");
            }
        });

        return sb.toString();
    }

    public static String measureValueToString(double value) {
        return Long.valueOf(Double.valueOf(value).longValue()).toString();
    }

    public static Long measureValueToLong(double value) {
        return Long.valueOf(Double.valueOf(value).longValue());
    }

    public static String prettyFormat(Meter meter) {

        return "";
    }

    public static void formatMeters(List<Meter> meters, boolean pretty) {
        meters.forEach(m -> {
            Logging.stat(format(m));
        });

        if (!pretty) {
            return;
        }

        if (!meters.isEmpty()) {
            AsciiTable at = new AsciiTable();
            at.getRenderer().setCWC(new CWC_LongestLine());
            at.getContext().setGridTheme(TA_GridThemes.NONE);

            at.addRow(MetricsPoint.getHeaderAsCollection());
            meters.forEach(m -> {
                MetricsPoint point = MetricsPoint.valueOf(m);
                at.addRow(point.getTextAsCollection());
            });
            Logging.stat("\n" + at.render());
        }
    }

    public static String getItem(Meter meter) {
        String item = meter.getId().getTag("apiName");
        if (!Strings.isNullOrEmpty(item)) {
            return item;
        }

        return Strings.nullToEmpty(meter.getId().getTag("item"));
    }
}
