package common.metrics.micrometer.registry;

import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.Tags;
import io.micrometer.core.instrument.binder.MeterBinder;
import io.micrometer.core.instrument.binder.jvm.JvmGcMetrics;
import io.micrometer.core.instrument.binder.jvm.JvmMemoryMetrics;
import io.micrometer.core.instrument.binder.jvm.JvmThreadMetrics;
import io.micrometer.core.instrument.binder.system.ProcessorMetrics;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Map;
import java.util.Objects;


public class SystemMetrics {
    private static Logger logger = LoggerFactory.getLogger(SystemMetrics.class);
    private IMetricsMeterRegistryConfig config;
    private Map<String, MeterBinder> binderMap = Maps.newHashMap();

    private long systemLatestReport = 0;

    public SystemMetrics() {
        this.config = new MetricsMeterRegistryConfig();
        addBind("jvm.memory", new JvmMemoryMetrics(Tags.of("_system_", "jvm.memory")));
        addBind("jvm.gc", new JvmGcMetrics(Tags.of("_system_", "jvm.gc")));
        addBind("jvm.thread", new JvmThreadMetrics(Tags.of("_system_", "jvm.thread")));
        addBind("processor", new ProcessorMetrics(Tags.of("_system_", "cpu.load")));
    }

    public void addBind(String key, MeterBinder binder) {
        if (!binderMap.containsKey(key)) {
            binderMap.putIfAbsent(key, binder);
            logger.info("Add Metrics bind:[{}]", key);
        } else {
            logger.info("Metrics bind already exists:[{}]", key);
        }
    }

    /**
     * 检查是否到系统监控上报时间
     * @return
     */
    public boolean checkAndSetReportStep() {
        int systemStep = config.getInt("metrics.system.report.step", 300);
        long current = System.currentTimeMillis() / 1000;
        if (current - systemLatestReport > systemStep) {
            systemLatestReport = current;
            return true;
        } else {
            return false;
        }
    }

    public void publish(MeterRegistry registry) {
        if (!checkAndSetReportStep()){
            return;
        }

        for (String key : binderMap.keySet()) {
            String configKey = String.format("%s.enable", key);
            if (config.getBoolean(configKey)) {
                MeterBinder binder = binderMap.get(key);
                if (Objects.nonNull(binder)) {
                    binder.bindTo(registry);
                }
            }
        }
    }

    private static class SystemMetricsHolder {
        private static SystemMetrics INSTANCE = new SystemMetrics();
    }

    public static SystemMetrics getInstance() {
        return SystemMetricsHolder.INSTANCE;
    }
}
