package common.metrics.micrometer.registry;

import com.google.common.base.Strings;
import common.config.tools.config.ConfigTools3;
import common.log.Logging;
import common.base.tools.sys.SystemTools;
import io.micrometer.core.instrument.*;
import io.micrometer.core.instrument.binder.jvm.JvmGcMetrics;
import io.micrometer.core.instrument.binder.jvm.JvmMemoryMetrics;
import io.micrometer.core.instrument.binder.jvm.JvmThreadMetrics;
import io.micrometer.core.instrument.binder.system.ProcessorMetrics;
import io.micrometer.core.instrument.step.StepMeterRegistry;



import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

public class MetricsMeterRegistry extends StepMeterRegistry {
    private IMetricsMeterRegistryConfig config;
    private MetricMeterSender meterSender;

    public MetricsMeterRegistry(IMetricsMeterRegistryConfig config, Clock clock) {
        super(config, clock);
        this.config = config;
        this.config().meterFilter(new MetricsMeterFilter(config));

        meterSender = new MetricMeterSender(config);

        //add common tag if report to remote
        if (this.config.enabled()) {
            this.config().commonTags("host", SystemTools.getHostname(), "appId", config.getAppId());
        }

        this.start(Executors.defaultThreadFactory());
        Metrics.addRegistry(this);
    }

    private void bindJvmMeter(IMetricsMeterRegistryConfig config) {

        if (config.getBoolean(MetricsMeterConstants.CONFIG_ENABLE_JVM_MEMORY)) {
            new JvmMemoryMetrics(Tags.of("system", "memory")).bindTo(this);
        }

        if (config.getBoolean(MetricsMeterConstants.CONFIG_ENABLE_JVM_GC)) {
            new JvmGcMetrics(Tags.of("system", "GC")).bindTo(this);
        }

        if (config.getBoolean(MetricsMeterConstants.CONFIG_ENABLE_JVM_THREAD)) {
            new JvmThreadMetrics(Tags.of("system", "thread")).bindTo(this);
        }

        if (config.getBoolean(MetricsMeterConstants.CONFIG_ENABLE_PROCESSOR)) {
            new ProcessorMetrics(Tags.of("system", "cpu")).bindTo(this);
        }
    }

    public MetricsMeterRegistry(IMetricsMeterRegistryConfig config) {
        this(config, Clock.SYSTEM);
    }

    @Override
    protected void publish() {
        try {
            //Add system metrics
            SystemMetrics.getInstance().publish(this);

            if (config.enabled()) {
                boolean pretty = config.getBoolean(MetricsMeterConstants.CONFIG_PRETTY_PRINT);

                //not system metric
                MetricsMeterFormat.formatMeters(getMeters().stream().filter(this::nonSystemMetric).collect(Collectors.toList()), pretty);

                //system metrics(jvm.memory/gc/threads/cpu load)
                MetricsMeterFormat.formatMeters(getMeters().stream().filter(this::isSystemMetric).collect(Collectors.toList()), pretty);

                if (config.enableToServer()) {
                    meterSender.sendReport(getMeters());
                }
            }
        } finally {
            //上报完之后清除meter
            clear();
        }
    }


    @Override
    protected TimeUnit getBaseTimeUnit() {
        return TimeUnit.MILLISECONDS;
    }

    private boolean isSystemMetric(Meter meter) {
        String value = meter.getId().getTag("_system_");
        return !Strings.isNullOrEmpty(value);
    }

    private boolean nonSystemMetric(Meter meter) {
        return !isSystemMetric(meter);
    }
}