package common.metrics.starter.filter;

import com.google.common.base.Joiner;
import com.google.common.base.Splitter;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Maps;
import common.base.tools.type.NumberTools;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.nio.charset.StandardCharsets;
import java.util.Base64;
import java.util.Map;

public class HttpTools {

    private static ImmutableList<String> ipHeaderNameList = ImmutableList.<String>builder()
            .add("x-forwarded-for")
            .add("CF-Connecting-IP")
            .add("Proxy-Client-IP")
            .add("WL-Proxy-Client-IP")

            .add("X-Forwarded-For")
            .add("HTTP_X_REAL_IP")
            .add("HTTP_CLIENT_IP")
            .add("HTTP_X_FORWARDED_FOR")

            .build();


    private static boolean isInvalidClientIp(String clientIp) {
        if (Strings.isNullOrEmpty(clientIp)) {
            return false;
        }

        if ("unknown".equalsIgnoreCase(clientIp)) {
            return false;
        }

        return true;
    }

    public static String getRemoteHost(HttpServletRequest request) {

        /*
           Get real client ip,support CloudFlare,Aliyun,AWS proxy
         */
        String remoteHost = null;
        for (String headerName : ipHeaderNameList) {
            remoteHost = request.getHeader(headerName);
            if (isInvalidClientIp(remoteHost)) {
                break;
            }
        }

        //CAN NOT get from header,get from request
        if (!isInvalidClientIp(remoteHost)) {
            remoteHost = request.getRemoteAddr();
        }

        //对于通过多个代理的情况，第一个IP为客户端真实IP,多个IP按照','分割
        if (!Strings.isNullOrEmpty(remoteHost) && remoteHost.length() > 15) { //"***.***.***.***".length() = 15
            if (remoteHost.indexOf(",") > 0) {
                remoteHost = remoteHost.substring(0, remoteHost.indexOf(","));
            }
        }

        return Strings.nullToEmpty(remoteHost);
    }


    public static String getHttpParameter(HttpServletRequest request, String key) {
        if (request == null) {
            return "";
        }

        if (Strings.isNullOrEmpty(key)) {
            return "";
        }

        return request.getParameter(key);
    }

    public static Long getLongParameter(HttpServletRequest request, String key) {
        String v = getHttpParameter(request, key);
        if (Strings.isNullOrEmpty(v)) {
            return null;
        }

        return Long.valueOf(v);
    }

    public static Long getLongAttr(HttpServletRequest request, String key) {
        return getLongAttr(request, key, null);
    }

    public static Long getLongAttr(HttpServletRequest request, String key, Long defaultValue) {
        Object v = request.getAttribute(key);
        if (v == null) {
            return defaultValue;
        }

        if (v instanceof Long) {
            return (Long) v;
        } else if (v instanceof String) {
            if (Strings.isNullOrEmpty((String) v)) {
                return defaultValue;
            }

            return Long.valueOf((String) v);
        } else {
            return defaultValue;
        }
    }

    public static int getIntAttr(HttpServletRequest request, String key, int defaultValue) {
        Object v = request.getAttribute(key);
        if (v == null) {
            return defaultValue;
        }

        if (v instanceof Integer) {
            return (int) v;
        }

        if (v instanceof String) {
            return NumberTools.str2int((String) v);
        }

        return defaultValue;
    }


    public static String getAttr(HttpServletRequest request, String key, String defaultValue) {
        Object v = request.getAttribute(key);
        if (v == null) {
            return defaultValue;
        }

        return (String) v;
    }

    public static void setAttrIfAbsent(HttpServletRequest request, String key, Object value) {
        Object v = request.getAttribute(key);
        if (v == null) {
            request.setAttribute(key, value);
        }
    }

    public static void setAttr(HttpServletRequest request, String key, Object value) {
        request.setAttribute(key, value);
    }


    public static String getApiPath(HttpServletRequest request) {
        return request.getRequestURI();
    }

    public static String getRequestURI(HttpServletRequest request) {
        String apiName = request.getRequestURI();
        apiName = apiName.substring(1).replace("/", "_");
        return apiName;
    }

    public static String getClientRegion(HttpServletRequest request) {
        return Strings.nullToEmpty(request.getHeader("CF-IPCountry"));
    }

    public static String getHeader(HttpServletRequest request,
                                   HttpServletResponse response,
                                   String name) {
        String value = request.getHeader(name);
        if (Strings.isNullOrEmpty(value)) {
            value = response.getHeader(name);
        }

        return value;
    }

    public static void setRequestId(HttpServletResponse response, String mac, String loginId, String reqId) {
        String reqIdStr = buildRequestId(mac, loginId, reqId);

        response.setHeader("REQID", reqIdStr);
    }

    public static String getRequestId(HttpServletResponse response) {
        return response.getHeader("REQID");
    }

    public static Map<String, String> getRequestMap(HttpServletResponse response) {
        String reqStr = getRequestId(response);
        if (Strings.isNullOrEmpty(reqStr)) {
            return Maps.newLinkedHashMap();
        }

        return parseRequestId(reqStr);
    }

    public static String buildRequestId(String mac, String loginId, String reqId) {
        Map<String, String> valueMap = Maps.newLinkedHashMap();
        if (!Strings.isNullOrEmpty(mac)) {
            valueMap.put("mac", mac);
        }

        if (!Strings.isNullOrEmpty(loginId)) {
            valueMap.put("login_id", loginId);
        }

        if (!Strings.isNullOrEmpty(reqId)) {
            valueMap.put("req_id", reqId);
        }

        String value = Joiner.on("::").withKeyValueSeparator("=").join(valueMap);
        return Base64.getUrlEncoder().encodeToString(value.getBytes(StandardCharsets.UTF_8));
    }

    public static Map<String, String> parseRequestId(String requestId) {
        if (Strings.isNullOrEmpty(requestId)) {
            return Maps.newLinkedHashMap();
        }

        String reqId = new String(Base64.getUrlDecoder().decode(requestId));
        Map<String, String> valueMap = Splitter.on("::").withKeyValueSeparator("=").split(reqId);
        return valueMap;
    }
}
